/*
 * OS/2 VT220 Emulation.
 *
 * Copyright (c) 1995-1996 Robert Muchsel, muchsel@acm.org
 *                                         (lifetime e-mail address)
 *
 * Permission is granted to use this code for non-commercial purposes.
 * This copyright notice must not be removed.
 *
 * NO WARRANTY!
 *
 * This code does NOT provide a complete VT 220 emulation. However, it
 * should understand all control sequences and key codes commonly found in
 * the /etc/termcap files, especially the Linux "console". The emulation
 * has been checked using the vttest utility and the ncurses samples.
 *
 * Not yet implemented (among others):
 *
 *   ESC #3   DECDHL  - double height line top half
 *   ESC #4   DECDHL  - double height line bottom half
 *   ESC #5   DECSWL  - single width line
 *   ESC #6   DECDWL  - double width line
 *
 *   ESC =    DECKPAM - keypad application mode
 *   ESC >    DECKPNM - keypad numeric mode
 *
 *   VT52 emulation
 *
 *   All sequences returning messages (<ENQ>, DSR, DA, DECREQTPARM, etc.):
 *
 *   ESC [R
 *   ESC [0c, Z
 *   ESC [0n, [3n, [5n, [6n
 *
 *
 * The emulation has no printing support. This is a feature and avoids
 * unwanted screen dumps. Use OS/2 PrtSc or Copy/Paste instead.
 *
 * The code has been tested against Linux, Sun and SGI termcaps/terminfos.
 *
 * History:
 *
 * 1995/09/09  Initial version
 * 1995/09/10  Made some changes so it passes the vttest
 * 1995/09/11  Added some Linux console ESC sequences
 * 1996/01/01  Added null character mapping (Linux), Linux key codes
 *
 */


/*
  DEBUG_VT220 prints all ESC codes and characters to stderr.
  Use ssh 2> logfile  (when using CMD)
   or ssh >&> logfile (when using 4OS2)
  to redirect the debug output.
 */

#undef DEBUG_VT220
/* #define DEBUG_VT220 /**/

#ifdef DEBUG_VT220
#include <stdio.h>
#endif

#include <stdlib.h>
#define INCL_VIO
#include <os2.h>
#include <sys/kbdscan.h>
#include "vt220.h"
#include "os2support.h"


/*
 * The ASCII control characters
 */
#define NUL   0
#define SOH   1
#define STX   2
#define ENQ   5
#define BEL   7
#define BS    8
#define HT    9
#define LF   10
#define VT   11
#define FF   12
#define CR   13
#define SO   14
#define SI   15
#define DLE  16
#define XON  17
#define XOFF 18
#define SYN  22
#define CAN  24
#define SUB  26
#define ESC  27
#define XFS  28
#define XGS  29
#define XRS  30
#define US   31
#define SP   32


/*
 * Colors
 */
#define COLOR_BLACK         0
#define COLOR_BLUE          1
#define COLOR_GREEN         2
#define COLOR_CYAN          3
#define COLOR_RED           4
#define COLOR_MAGENTA       5
#define COLOR_YELLOW        6
#define COLOR_WHITE         7
#define COLOR_BRIGHT        8


/*
 * Emulator state variables
 */
static int  xpos  = 1;
static int  ypos  = 1;
static int  topmargin = 1;
static int  botmargin = 25;

static int  state = 0;
static char esc_buffer[255];

#define MAXW 400  /* This is the maximum screen width, 400 should suffice */
                  /* The maximum screen size is MAXW*MAXW */
static char tabs[MAXW];

static int  screen_width  = 80;
static int  screen_height = 25;

#define ATTR_DEFAULT   COLOR_WHITE
#define ATTR_UNDERLINE COLOR_GREEN+COLOR_BRIGHT
static int  attr           = ATTR_DEFAULT;
static int  attr_default   = ATTR_DEFAULT;
static int  attr_underline = ATTR_UNDERLINE;

/*
 * Color mapping
 */
static int  colors[8] = {
                          COLOR_BLACK,
                          COLOR_RED,
                          COLOR_GREEN,
                          COLOR_YELLOW,
                          COLOR_BLUE,
                          COLOR_MAGENTA,
                          COLOR_CYAN,
                          COLOR_WHITE
                        };

/*
 * VT100 graphics character mapping
 */
static int  vt_graph[] = {
  0x04, 0xB0, 0x1A, 0x17, 0x1B, 0x19, 0xF8, 0xF1, 0x15, 0x12,
  0xD9, 0xBF, 0xDA, 0xC0, 0xC5, 0xC4, 0xC4, 0xC4, 0xC4, 0xC4,
  0xC3, 0xB4, 0xC1, 0xC2, 0xB3, 0xF3, 0xF2, 0xE3, 0x9D, 0x9C,
  0xFA
};

static int  blink  = 0;
static int  bold   = 0;
static int  invis  = 0;
static int  underl = 0;
static int  revers = 0;

static int  saved_x      = 1;
static int  saved_y      = 1;
static int  saved_attr   = ATTR_DEFAULT;
static int  saved_blink  = 0;
static int  saved_bold   = 0;
static int  saved_invis  = 0;
static int  saved_underl = 0;
static int  saved_revers = 0;
static int  saved_g0     = 0;
static int  saved_g1     = 1;
static int  saved_chrset = 0;

static int  auto_wrap   = 1;
static int  insert_mode = 0;
static int  appl_mode   = 0;
static int  cursor_vis  = 1;
static int  col80       = 0;
static int  scr_reverse = 0;
static int  relative    = 0;
static int  crlf        = 1;
static int  graph_g0    = 0;
static int  graph_g1    = 1;
static int  charset     = 0;

VIOCURSORINFO Cursor;

/*
 * You must NOT remove the following notice. It will be compiled into
 * your executable file (if the linker is not smart enough to remove it...)
 */
char copyright[] = "Portions Copyright (c) 1995-1996 Robert Muchsel\r\n";


/*
 * Convention: All exported functions are called vt220_...
 */


/*
 * Set default values (don't move the cursor)
 */
void defaults(void)
{
  int i;

  state = 0;
  xpos  = 1;
  ypos  = 1;

  memset(tabs, 0, sizeof(tabs));
  for (i = 1; i < MAXW; i += 8)
    tabs[i] = 1;

  attr           = ATTR_DEFAULT;
  attr_default   = ATTR_DEFAULT;
  attr_underline = ATTR_UNDERLINE;

  blink  =
  bold   =
  invis  =
  underl =
  revers = 0;

  auto_wrap   = 1;
  insert_mode = 0;
  appl_mode   = 0;
  cursor_vis  = 1;
  col80       = 0;
  scr_reverse = 0;
  relative    = 0;
  crlf        = 1;
  graph_g0    = 0;
  graph_g1    = 1;
  charset     = 0;

  topmargin = 1;
  botmargin = screen_height;
}


/*
 * Reset the emulator (call this function at startup)
 */
void vt220_reset(void)
{
  int i;

  viogetmode(&screen_width, &screen_height, &i, &i);
  defaults();

  VioGetCurPos((PUSHORT) &ypos, (PUSHORT) &xpos, 0);
  xpos++; /* OS/2 Vio coordinates are 0-based */
  ypos++;

  /* Save cursor */
  VioGetCurType(&Cursor, 0);
}


/*
 * Beep
 */
void beep(void)
{
  DosBeep(1660, 25);
}


/*
 * Disable (hide) cursor
 */
void cursor_off(void)
{
  VIOCURSORINFO CursorData;

  CursorData.yStart =  0;
  CursorData.cEnd   =  0;
  CursorData.cx     =  0;
  CursorData.attr   = -1;
  VioSetCurType(&CursorData, 0);        /* Hide cursor */
}

/*
 * Enable (show) cursor
 */
void cursor_on(void)
{
  VioSetCurType(&Cursor, 0);
}


/*
 * Move cursor, clip. Clip to margins if 'rel_y' is true
 */
void inline moveto(int rel_y, int x, int y)
{
  if (x < 1)
    x = 1;
  if (x > screen_width)
    x = screen_width;
  if (rel_y)
    {
      if (y < topmargin)
        y = topmargin;
      if (y > botmargin)
        y = botmargin;
    }
  else
    {
      if (y < 1)
        y = 1;
      if (y > screen_height)
        y = screen_height;
    }

  xpos = x;
  ypos = y;

  if (cursor_vis)
    VioSetCurPos(y-1, x-1, 0);

  col80 = 0;
}


/*
 * Clear to end of line
 */
void clear_eol(void)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioWrtNCell(Cell, screen_width+1 - xpos, ypos-1, xpos-1, 0);
}


/*
 * Clear to beginning of line
 */
void clear_bol(void)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioWrtNCell(Cell, xpos, ypos-1, 0, 0);
}


/*
 * Clear line
 */
void clear_line(void)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioWrtNCell(Cell, screen_width, ypos-1, 0, 0);
}


/*
 * Clear to end of screen
 */
void clear_eos(void)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioWrtNCell(Cell,
    screen_width*screen_height - (ypos-1)*screen_width - (xpos-1),
    ypos-1, xpos-1, 0);
}


/*
 * Clear to beginning of screen
 */
void clear_bos(void)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioWrtNCell(Cell, (ypos-1)*screen_width + xpos, 0, 0, 0);
}


/*
 * Clear screen
 */
void clear_scr(char c)
{
  char Cell[2];

  Cell[0] = c;
  Cell[1] = attr_default;
  VioWrtNCell(Cell, screen_width*screen_height, 0, 0, 0);
}


/*
 * Flip a color.
 */
int inline flip(int color)
{
  return (color & 0x88) |
        ((color & 0x70) >> 4) |
        ((color & 0x07) << 4);
}


/*
 * Reverse screen
 */
void reverse_scr()
{
  char Cell[2 * MAXW*MAXW];
  int  Length;

  /* Flip all colors */
  attr           = flip(attr);
  attr_default   = flip(attr_default);
  attr_underline = flip(attr_underline);

  /* Flip the screen */
  Length = 2 * screen_width*screen_height;
  VioReadCellStr(Cell, (PUSHORT) &Length, 0, 0, 0);
  for (Length = 0; Length < screen_width*screen_height; Length++)
    Cell[2*Length+1] = flip(Cell[2*Length+1]);
  VioWrtCellStr(Cell, 2*Length, 0, 0, 0);
}


/*
 * Scroll (direction -1 = UP, +1 = DOWN)
 */
void scroll(int direction, int topy, int boty, int numlines)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  if (direction < 0)
    VioScrollUp(topy-1, 0, boty-1, screen_width-1, numlines, Cell, 0);
  else
    VioScrollDn(topy-1, 0, boty-1, screen_width-1, numlines, Cell, 0);
}


/*
 * Write character to screen on 'xpos'/'ypos' using 'attr'
 * Take care of the rightmost column (col80) and autowrap
 */
void screen_write(unsigned char c)
{
  char Cell[2];
  int wrap80 = 0;

  switch (c)
    {
      case LF: /* advance one line, scroll if needed */
               if (ypos < botmargin)
                 ypos++;
               else
                 scroll(-1, topmargin, botmargin, 1);
      break;

      case CR:
               xpos = 1;
      break;

      case BS:
               xpos--;
      break;

      default:
               if (col80 && (xpos == screen_width))
                 if (auto_wrap) /* wrap line if longer than screenwidth chars */
                   {
                     if (ypos < botmargin)
                       ypos++;
                     else
                       scroll(-1, topmargin, botmargin, 1);
                     xpos = 1;
                       moveto(0, xpos, ypos);
                   }
                 else
                   return; /* forget the character */

               if (invis)
                 c = SP;
               else
                 if ((c >= 96) && (c <= 126) &&
                    ((charset && graph_g1) || (!charset && graph_g0)))
                   c = vt_graph[c - 96];

               if (insert_mode)
                 {
                   Cell[0] = c;
                   Cell[1] = attr;
                   VioScrollRt(ypos-1, xpos-1, ypos-1, screen_width-1, 1, Cell, 0);
                 }
               else
                 VioWrtCharStrAtt((PCH) &c, 1, ypos-1, xpos-1, (PBYTE) &attr, 0);

               xpos++;
               wrap80 = xpos > screen_width;

      break;
    } /* switch */

  moveto(0, xpos, ypos);
  col80 = wrap80;
}


/*
 * Write a string to screen on 'xpos'/'ypos' using 'attr'.
 * This function does not wrap nor process control characters.
 */
void inline screen_fastwrite(int len, char *str)
{
  char Cell[2];
  unsigned char *s;

  s = str;

  /* process characters */
  while (*s)
    {
      if (invis)
        *s = SP;
      else
        if ((*s >= 96) && (*s <= 126) &&
          ((charset && graph_g1) || (!charset && graph_g0)))
          *s = vt_graph[*s - 96];

      s++;
    }

  if (insert_mode)
    {
      Cell[0] = SP;
      Cell[1] = attr_default;

      VioScrollRt(ypos-1, xpos-1, ypos-1, screen_width-1, len, Cell, 0);
    }

  VioWrtCharStrAtt((PCH) str, len, ypos-1, xpos-1, (PBYTE) &attr, 0);

  moveto(0,xpos+len,ypos);
}


/*
 * Delete character on screen
 */
void screen_del(int numchars)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioScrollLf(ypos-1, xpos-1, ypos-1, screen_width-1, numchars, Cell, 0);
}


/*
 * Insert character on screen
 */
void screen_ins(int numchars)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioScrollRt(ypos-1, xpos-1, ypos-1, screen_width-1, numchars, Cell, 0);
}


/*
 * Overwrite character on screen
 */
void screen_over(int numchars)
{
  char Cell[2];

  Cell[0] = SP;
  Cell[1] = attr_default;
  VioWrtNCell(Cell, numchars+1 - xpos, ypos-1, xpos-1, 0);
}


/*
 * Save cursor position and attributes
 */
void save_cursor(void)
{
  saved_x      = xpos;
  saved_y      = ypos;
  saved_attr   = attr;
  saved_blink  = blink;
  saved_bold   = bold;
  saved_invis  = invis;
  saved_underl = underl;
  saved_revers = revers;
  saved_g0     = graph_g0;
  saved_g1     = graph_g1;
  saved_chrset = charset;
}


/*
 * Restore cursor position and attributes
 */
void restore_cursor(void)
{
  xpos     = saved_x;
  ypos     = saved_y;
  attr     = saved_attr;
  blink    = saved_blink;
  bold     = saved_bold;
  invis    = saved_invis;
  underl   = saved_underl;
  revers   = saved_revers;
  graph_g0 = saved_g0;
  graph_g1 = saved_g1;
  charset  = saved_chrset;

  moveto(0, xpos, ypos);
}


/*
 * Handle ESC sequences
 */
void esc_sequence()
{
#ifdef DEBUG_VT220
  fprintf(stderr, "ESC: %.*s (%c)\n", state-1, &esc_buffer[1],
          esc_buffer[1]);
#endif

  esc_buffer[state] = 0;

  /* first character is action code */
  switch (esc_buffer[1])
    {
      case '#': /* ESC # */
                switch (esc_buffer[2])
                  {
                    case '8': /* DECALN - self test */
                              clear_scr('E');
                              topmargin = 1;
                              botmargin = screen_height;
                              moveto(0,1,1);
                    break;

                    default:
                    break;
                  }
      break;

      case '7': /* DECSC - save cursor */
                save_cursor();
      break;

      case '8': /* DECRC - restore cursor */
                restore_cursor();
      break;

      case 'D': /* IND - index */
                if (ypos < botmargin)
                  moveto(0, xpos, ypos+1);
                else
                  scroll(-1, topmargin, botmargin, 1);
                col80 = 0;
      break;

      case 'E': /* NEL - next line */
                screen_write(CR);
                screen_write(LF);
      break;

      case 'H': /* HTS - horizontal tabulation set */
                tabs[xpos] = 1;
      break;

      case 'M': /* RI - reverse index */
                if (ypos > topmargin)
                  moveto(0, xpos, ypos-1);
                else
                  scroll(1, topmargin, botmargin, 1);
                col80 = 0;
      break;

      case 'c': /* ??? - reset */
                defaults();
                clear_scr(SP);
                moveto(0,1,1);
      break;

      case '(': /* SCS - select GO character set */
      case ')': /* SCS - select G1 character set */
                switch (esc_buffer[2])
                  {
                    case '0': /* VT graphics */
                              if (esc_buffer[1] == '(')
                                graph_g0 = 1;
                              else
                                graph_g1 = 1;
                    break;

                    case 'A': /* UK ASCII */
                    case 'B': /* US ASCII */
                    case 'K': /* user mapping */
                    case 'U': /* null mapping */
                    case '1':
                    case '2':
                              if (esc_buffer[1] == '(')
                                graph_g0 = 0;
                              else
                                graph_g1 = 0;
                    break;

                    default:  /* ignore */
                    break;
                  }
      break;

      default:  /* just ignore */
      break;
    }
}


/*
 * Handle ESC CSI screen control sequences
 */
void ctrl_sequence()
{
  int arg[200];
  int na = 0;
  char s[20];
  int escpos, spos, i, private, set;

  /* the buffer contains the complete ESC sequence, ESC[..;..;.X */

  esc_buffer[state] = 0;

  /* are there arguments? */
  escpos  = 2;
  private = 0;

  while (escpos < state)
    {
      spos = 0;
      if (esc_buffer[escpos] == '?')
        {
          private = 1;
          escpos++;
        }

      while ((esc_buffer[escpos] >= '0') && (esc_buffer[escpos] <= '9'))
        {
          s[spos] = esc_buffer[escpos];
          spos++;
          escpos++;
        }
      s[spos] = 0;
      if (*s)
        {
#ifdef DEBUG_VT220
          fprintf(stderr,"arg: %s\n", s);
#endif
          if (!private)
            arg[na] = atoi(s);
          else
            arg[na] = -1*atoi(s);
          na++;
        }
      escpos++;
    }

  if (na == 0)
    arg[0] = 1; /* default in many cases */
  if (na <= 1)
    arg[1] = 1;

#ifdef DEBUG_VT220
  fprintf(stderr, "ESC CSI: %.*s (%c)\n", state-1, &esc_buffer[1],
          esc_buffer[state-1]);
#endif

  /* last character contains action code */
  switch (esc_buffer[state-1])
    {
      case 'A': /* CUU - cursor up */
                if (arg[0] == 0)
                  arg[0] = 1;
                moveto(relative,xpos,ypos-arg[0]);
      break;

      case 'e':
      case 'B': /* CUD - cursor down */
                if (arg[0] == 0)
                  arg[0] = 1;
                moveto(relative,xpos,ypos+arg[0]);
      break;

      case 'a':
      case 'C': /* CUF - cursor forward */
                if (arg[0] == 0)
                  arg[0] = 1;
                moveto(0,xpos+arg[0],ypos);
      break;

      case 'D': /* CUB - cursor backward */
                if (arg[0] == 0)
                  arg[0] = 1;
                moveto(0,xpos-arg[0],ypos);
      break;

      case 'E': /* ??? - cursor down, column 1 */
                if (arg[0] == 0)
                  arg[0] = 1;
                moveto(relative,1,ypos+arg[0]);
      break;

      case 'F': /* ??? - cursor up, column 1 */
                if (arg[0] == 0)
                  arg[0] = 1;
                moveto(relative,1,ypos-arg[0]);
      break;

      case 'J': /* ED - erase in display */
                if ((na == 0) || (arg[0] == 0))
                  clear_eos();
                else
                  if (arg[0] == 1)
                    clear_bos();
                  else
                    if (arg[0] == 2)
                      clear_scr(SP);
      break;

      case 'K': /* EL - erase in line */
                if ((na == 0) || (arg[0] == 0))
                  clear_eol();
                else
                  if (arg[0] == 1)
                    clear_bol();
                  else
                    if (arg[0] == 2)
                      clear_line();
      break;

      case 'f': /* HVP - horizontal and vertical position */
      case 'H': /* CUP - cursor position */
                moveto(relative, arg[1], arg[0] + (relative ? topmargin-1 : 0));
      break;

      case 'G': /* CHA - cursor character absolute */
      case '`': /* HPA - character position absolute */
                moveto(0, arg[0], ypos);
      break;

      case 'd': /* VPA - Line position absolute */
                moveto(0, xpos, arg[0]);
      break;

      case 'g': /* TBC - tabulation clear */
                if ((na == 0) || (arg[0] == 0))
                  tabs[xpos] = 0;
                else
                  if (arg[0] == 3) /* clear all */
                    memset(tabs, 0, sizeof(tabs));
      break;

      case 'h': /* SM - set mode */
      case 'l': /* RM - reset mode */
                set = esc_buffer[state-1] == 'h';

                for (i = 0; i < na; i++)
                  {
                    switch (arg[i])
                      {
                        case  -1: /* DECCKM  - cursor keys application mode */
                                  appl_mode   = set;
                        break;

                        case  -5: /* DECSCNM - reverse screen */
                                  if (scr_reverse ^ set)
                                    {
                                      scr_reverse = set;
                                      reverse_scr();
                                    }
                        break;

                        case  -6: /* DECOM   - relative origin */
                                  relative    = set;
                                  moveto(relative, 1, 1);
                        break;

                        case  -7: /* DECAWM  - auto wrap */
                                  auto_wrap   = set;
                        break;

                        case -25: /* DECTCEM - cursor on */
                                  if (set)
                                    {
                                      cursor_on();
                                      cursor_vis = 1;
                                      moveto(0, xpos, ypos);
                                    }
                                  else
                                    {
                                      cursor_off();
                                      cursor_vis = 0;
                                    }
                        break;

                        case   4: /* ANSI insert mode */
                                  insert_mode = set;
                        break;

                        case  20: /* LNM - linefeed / newline mode */
                                  crlf        = set;
                        break;

                        default:  /* ignore all others */
                        break;
                      }
                  }
      break;

      case 'L': /* IL - insert line */
                for (i = 1; i <= arg[0]; i++)
                  scroll(1, ypos, botmargin, 1);
      break;

      case 'M': /* DL - delete line */
                for (i = 1; i <= arg[0]; ++i)
                  scroll(-1, ypos, botmargin, 1);
      break;

      case 'm': /* SGR - select graphic rendition */
                if (na == 0)
                  {
                    attr   = attr_default;

                    blink  =
                    bold   =
                    invis  =
                    underl =
                    revers = 0;
                  }
                else
                  for (i = 0; i < na; i++)
                    {
                      switch (arg[i])
                        {
                          case  0: /* reset */
                                   attr   = attr_default;

                                   blink  =
                                   bold   =
                                   invis  =
                                   underl =
                                   revers = 0;
                          break;

                          case 21: /* bold off */
                          case 22:
                          case  1: /* bold */
                          case  2:
                                   if (bold ^ (arg[i] < 20))
                                     {
                                       bold   = arg[i] < 20;
                                       attr  ^= 8;
                                     }
                          break;

                          case 24: /* underline off */
                          case  4: /* underline */
                                   if (underl ^ (arg[i] == 4))
                                     {
                                       underl = arg[i] == 4;
                                       if (underl)
                                         attr = attr_underline;
                                       else
                                         attr = attr_default;
                                       if (bold)
                                         attr ^= 8;
                                       if (blink)
                                         attr ^= 0x80;
                                       if (revers) /* swap colors */
                                         attr = flip(attr);
                                     }
                          break;

                          case 25: /* blink off */
                          case  5: /* blink */
                                   if (blink ^ (arg[i] == 5))
                                     {
                                       blink = arg[i] == 5;
                                       attr ^= 0x80;
                                     }
                          break;

                          case 27: /* reverse off */
                          case  7: /* reverse */
                                   if (revers ^ (arg[i] == 7))
                                     {
                                       revers = arg[i] == 7;
                                       attr   = flip(attr);
                                     }
                          break;

                          case 28: /* invisible off */
                          case  8: /* invisible */
                                   invis = arg[i] == 8;
                          break;

                          case 30:
                          case 31:
                          case 32:
                          case 33:
                          case 34:
                          case 35:
                          case 36:
                          case 37: /* foreground colors */
                                   if (!scr_reverse)
                                     {
                                       attr &= 0xF8;
                                       attr |= colors[arg[i] - 30];
                                     }
                                   else
                                     {
                                       attr &= 0x8F;
                                       attr |= colors[arg[i] - 30] << 4;
                                     }
                          break;

                          case 40:
                          case 41:
                          case 42:
                          case 43:
                          case 44:
                          case 45:
                          case 46:
                          case 47: /* background colors */
                                   if (!scr_reverse)
                                     {
                                       attr &= 0x8F;
                                       attr |= colors[arg[i] - 40] << 4;
                                     }
                                   else
                                     {
                                       attr &= 0xF8;
                                       attr |= colors[arg[i] - 40];
                                     }
                          break;

                          default: /* ignore */
                          break;
                        }
                    }
      break;

      case 'r': /* DECSTBM - set top and bottom margins */
                i = na >= 2 ? arg[1] : screen_height;
                if ((arg[0] >= 1) && (arg[0] < i) && (i <= screen_height))
                  {
                    topmargin = arg[0];
                    botmargin = i;
                  }
                moveto(relative, 1, 1);
      break;

      case 'P': /* DCH - delete character */
                if (arg[0] > screen_width + 1 - xpos)
                  arg[0] = screen_width + 1 - xpos;
                screen_del(arg[0]);
      break;

      case '@': /* ICH - insert character */
                if (arg[0] > screen_width + 1 - xpos)
                  arg[0] = screen_width + 1 - xpos;
                screen_ins(arg[0]);
      break;

      case 'X': /* ECH - erase character */
                if (arg[0] > screen_width + 1 - xpos)
                  arg[0] = screen_width + 1 - xpos;
                screen_over(arg[0]);
      break;

      case 's': /* ??? - save cursor and attributes */
                save_cursor();
      break;

      case 'u': /* ??? - restore cursor and attributes */
                restore_cursor();
      break;

      default:  /* ignore */
      break;
    }
}


/*
 * Advance the source pointer, read from buffered esc sequence if
 * needed, else from the buffer
 */
void inline advance(char **pcptr, int *remembered_chars, char *buffer)
{
  if (*remembered_chars > 0)
    {
      (*remembered_chars)--;
      if (*remembered_chars > 0)
        (*pcptr)++;
      else
        *pcptr = buffer;
    }
  else
    (*pcptr)++;
}


/*
 * Write a screen sequence and process characters. If an escape sequence
 * is detected, set the state variable and remember the sequence. If the
 * sequence isn't finished, the sequence is re-parsed from the esc_buffer
 * when the function is called again.
 */
void vt220_write(char *buffer, int len)
{
  char *bptr, *cptr;
  int pos, x;
  int remembered_chars;
  char buf[MAXW];

  remembered_chars = state;
  if (remembered_chars > 0)
    cptr = esc_buffer;
  else
    cptr = buffer;

  pos = -remembered_chars;

  while (pos < len)
    {
      switch (*cptr)
        {
          case ESC :  /* ESC sequence */
restart_ESC:
                      state = 0;
                      esc_buffer[state] = *cptr;
                      state++;
restart_ESC1:
                      advance(&cptr, &remembered_chars, buffer);
                      pos++;
                      if (pos >= len)
                        return;

                      esc_buffer[state] = *cptr;
                      state++;

                      switch (*cptr)
                        {
                          case CAN :  /* cancel sequence */
                          case SUB :
                          case BS :
                                      goto quit_ESC;
                          break;

                          case ESC :  /* restart sequence */
                                      goto restart_ESC;
                          break;

                          case '[' :  /* CSI screen control sequence */
                                      advance(&cptr, &remembered_chars, buffer);
                                      pos++;
                                      if (pos >= len)
                                        return;

                                      esc_buffer[state] = *cptr;
                                      state++;

                                      if (*cptr == ESC)
                                        goto restart_ESC;
                                      if ((*cptr == CAN) || (*cptr == SUB))
                                        goto quit_ESC;
                                      if (*cptr == BS)
                                        { state = 1; goto restart_ESC1; }

                                      while ((*cptr < '@') || (*cptr > '~'))
                                        {
                                          if (*cptr == ESC)
                                            goto restart_ESC;
                                          if ((*cptr == CAN) || (*cptr == SUB))
                                            goto quit_ESC;
                                          if (*cptr == BS)
                                            state -= 2;

                                          advance(&cptr, &remembered_chars, buffer);
                                          pos++;
                                          if (pos >= len)
                                            return;

                                          esc_buffer[state] = *cptr;
                                          state++;
                                        }
                                      ctrl_sequence();
                          break;

                          case '_' :  /* application program command */
                          case 'P' :  /* device control string */
                          case '^' :  /* privacy message */
                          case ']' :  /* operating system command */
                                      /* eat string until ESC \ */
                                      while (1)
                                        {
                                          advance(&cptr, &remembered_chars, buffer);
                                          pos++;
                                          if (pos >= len)
                                            return;

                                          if ((*cptr == '\\') && (esc_buffer[state] == ESC))
                                            break;

                                          esc_buffer[state] = *cptr;
                                          state++;
                                        }
                          break;

                          default :   /* escape sequence */
                                      while ((*cptr < '0') || (*cptr > '~'))
                                        {
                                          if (*cptr == ESC)
                                            goto restart_ESC;
                                          if ((*cptr == CAN) || (*cptr == SUB))
                                            goto quit_ESC;
                                          if (*cptr == BS)
                                            state -= 2;

                                          advance(&cptr, &remembered_chars, buffer);
                                          pos++;
                                          if (pos >= len)
                                            return;

                                          esc_buffer[state] = *cptr;
                                          state++;
                                        }
                                      esc_sequence();
                          break;
                        }
quit_ESC:
          break;

          case ENQ :  /* ignore */
          break;

          case HT :   /* go to next tab */
#ifdef DEBUG_VT220
                      fprintf(stderr,"[%c]", *cptr); fflush(stderr);
#endif
                      xpos++;
                      while ((xpos < screen_width) && !tabs[xpos])
                        xpos++;
                      moveto(0, xpos, ypos);
          break;

          case BEL :  /* beep */
                      beep();
          break;

          case VT :   /* simulate new line */
          case FF :
          case LF :
#ifdef DEBUG_VT220
                      fprintf(stderr,"[%c]", *cptr); fflush(stderr);
#endif
                      screen_write(LF);

                      if (crlf)
                        screen_write(CR);
          break;

          case SO :   /* ^N - select character set 1 */
          case SI :   /* ^O - select character set 0 */
#ifdef DEBUG_VT220
                      fprintf(stderr,"[%s]\n", *cptr==SO?"SO":"SI"); fflush(stderr);
#endif
                      charset = (*cptr) == SO;
          break;

          default :   /* all other characters */

                      /* To speed up the display, write several characters
                       * at a time (no wrap or other special processing,
                       * default attribute)
                       */
                      if ((*cptr >= SP) && (*cptr <= '~')
                       && (xpos < screen_width) && (pos < len))
                        {
                          x    = xpos;
                          bptr = buf;

                          while ((*cptr >= SP) && (*cptr <= '~')
                              && (x < screen_width) && (pos < len))
                            {
                              *bptr = *cptr;

                              advance(&cptr, &remembered_chars, buffer);
                              pos++;
                              bptr++;
                              x++;
                            }
                          *bptr = 0;

#ifdef DEBUG_VT220
                          fprintf(stderr,"%s", buf); fflush(stderr);
#endif

                          screen_fastwrite(x-xpos, buf);

                          continue; /* cptr has been advanced already */
                        }
                        else        /* normal screen write */
                          {
#ifdef DEBUG_VT220
                            fprintf(stderr,"%c", *cptr); fflush(stderr);
#endif
                            screen_write(*cptr);
                          }
          break;
        }

      advance(&cptr, &remembered_chars, buffer);
      pos++;
    }

  state = 0;
}


/*
 * Translate PC scan codes to VT220 ESC sequences
 *
 * This function discards characters if the buffer would overflow,
 * because input comes from a terminal and the user can be expected
 * not to type 32768 characters at a time.
 *
 * The F* function keys return Linux values. Shift-F* returns the
 * standard key code.
 *
 * This function MUST work correctly without prior initialization
 * by vt220_reset().
 */
void vt220_translate(char *buffer, int *len)
{
  char tmp[32768];
  unsigned char *sptr, *dptr;
  int left;

  memcpy(tmp, buffer, *len);

  sptr = tmp;
  dptr = buffer;
  left = *len;
  *len = 0;

  while (left > 0)
    {
      if (*sptr == 0)   /* found PC key scan code */
        {
          char *sequence;

          sptr++;
          left--;

          switch (*sptr)
            {
              case K_UP:
                        sequence = appl_mode ? "\eOA" : "\e[A";
              break;

              case K_DOWN:
                        sequence = appl_mode ? "\eOB" : "\e[B";
              break;

              case K_RIGHT:
                        sequence = appl_mode ? "\eOC" : "\e[C";
              break;

              case K_LEFT:
                        sequence = appl_mode ? "\eOD" : "\e[D";
              break;

              case K_SHIFT_F1:
                        sequence = "\eOP";
              break;

              case K_SHIFT_F2:
                        sequence = "\eOQ";
              break;

              case K_SHIFT_F3:
                        sequence = "\eOR";
              break;

              case K_SHIFT_F4:
                        sequence = "\eOS";
              break;

              case K_F1:
                        sequence = "\e[[A";
              break;

              case K_F2:
                        sequence = "\e[[B";
              break;

              case K_F3:
                        sequence = "\e[[C";
              break;

              case K_F4:
                        sequence = "\e[[D";
              break;

              case K_F5:
                        sequence = "\e[[E";
              break;

              case K_F6:
                        sequence = "\e[17~";
              break;

              case K_F7:
                        sequence = "\e[18~";
              break;

              case K_F8:
                        sequence = "\e[19~";
              break;

              case K_F9:
                        sequence = "\e[20~";
              break;

              case K_F10:
                        sequence = "\e[21~";
              break;

              case K_F11:
                        sequence = "\e[23~";
              break;

              case K_F12:
                        sequence = "\e[24~";
              break;

              case K_HOME:
                        sequence = "\e[1~";
              break;

              case K_INS:
                        sequence = "\e[2~";
              break;

              case K_DEL:
                        sequence = "\e[3~";
              break;

              case K_END:
                        sequence = "\e[4~";
              break;

              case K_PAGEUP:
                        sequence = "\e[5~";
              break;

              case K_PAGEDOWN:
                        sequence = "\e[6~";
              break;

              default:  /* ignore all others */
                        sequence = "";
              break;
            }

          while (*sequence)
            {
              *dptr = *sequence;
              (*len)++;
              if (*len == sizeof(tmp)) /* ungraceful overflow handling */
                return;
              dptr++;
              sequence++;
            }
        }
      else              /* normal character, just append it */
        {
          *dptr = *sptr;
          (*len)++;
          if (*len == sizeof(tmp))  /* ungraceful overflow handling */
            return;
          dptr++;
        }

      sptr++;
      left--;
    }
}
