/************************************************************************/
/*									*/
/*	PORTFNS.C	"Port Functions"				*/
/*									*/
/*   This file contains the functions that actually read and write	*/
/* bytes to and from the port.  These functions should be called when-	*/
/* ever communication through the port to the Initializer or CRYPTOCard	*/
/* is desired.  Also found herein is a function that initializes the	*/
/* settings of the port to that required by the Initializer.		*/
/* Dale Darling - May 8th, 1991.					*/
/*									*/
/*   (C) - 1991 CRYPTOCard Corporation.  All rights reserved.		*/
/*									*/
/************************************************************************/


/************************************************************************/
/* Header files:							*/
/************************************************************************/
#include <dos.h>
#include <time.h>
#include "intlzr.h"


/************************************************************************/
void InitPort (int Port)
/************************************************************************/
/* This function initializes the port specified in the parameters for	*/
/* communication with the Initializer.  The Initializer requires the 	*/
/* following setup:							*/
/*   8 Data Bits, 1 Stop Bit, No Parity, and 4800 Baud.			*/
/************************************************************************/
{					/* Here we go...		*/
  outportb (Port, 0);			/* Disable all interrupts	*/
  outportb (Port+LCR, 0x80);		/* Divisor Latch Access Bit hi	*/
/* CHECK BELOW!!!!  It says in the specs it should be 4800 Baud.	*/
  outportb (Port+DLL, 0x30);		/* Set the BAUD rate to 2400	*/
  outportb (Port+DLM, 0x00);		/* Complete the BAUD setting	*/
  outportb (Port+LCR, 0x03);		/* Reset DLAB.			*/
  outportb (Port+MCR, 0x01);		/* Put RTS in the data position	*/
}					/* That's it that's all		*/
/************************************************************************/


/************************************************************************/
int SendByte (byte Byte, int Port)
/************************************************************************/
/* This function sends the byte out the serial port specified in the	*/
/* parameters.  It waits for THR to empty before sending the byte.	*/
/* If the byte was sent successfully, the function returns the byte.	*/
/* Otherwise there was a timeout waiting for THR to empty, and the 	*/
/* function returns NOT Byte (!Byte).						*/
/* Dale Darling - May 8th, 1991.					*/
/************************************************************************/
{
  if (!WaitOnSentByte (Port))		/* There's a byte that won't go	*/
    return (!Byte);			/* Inform caller of error	*/
  outportb (Port+THR, Byte);		/* Send out the byte		*/
  if (!WaitOnSentByte (Port))		/* Our byte won't leave		*/
    return (!Byte);			/* Again return a bad value	*/
  return (Byte);			/* Otherwise, it was a success	*/
}
/************************************************************************/



/************************************************************************/
int WaitOnSentByte (int Port)
/************************************************************************/
/* This function waits for the Transmit Empty AND the Transmit Holding	*/
/* Register Empty bits of the Line Status Register to both go high	*/
/* before returning.  If TM_OUT_MAX seconds pass before this is true,	*/
/* then the function returns FALSE (0), otherwise it returns TRUE (1).	*/
/* In other words, it waits for any bytes transmitted to go completely.	*/
/* The parameter is the port in which to wait on.			*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  time_t BeginTime, CurTime;			/* Timer variables.	*/
  BeginTime = time (&BeginTime);		/* Set up timeout var	*/
  while ( (inportb (Port+LSR) & (THRE|TE)) != (THRE|TE) )
    if (difftime (time (&CurTime), BeginTime) > TM_OUT_MAX)  /* TIMEOUT	*/
      return (FALSE);				/* Return a bad value	*/
  return (TRUE);				/* All clean		*/
}
/************************************************************************/


/************************************************************************/
int GetByte (int Port)
/************************************************************************/
/* This function reads a byte from the serial port indicated in the	*/
/* parameter.  It waits a maximum of TM_OUT_MAX seconds before timing	*/
/* out and returning a -1 (Note that a FALSE should not be returned,	*/
/* as this is defined as 0, which is a valid byte. "-1" as an int could	*/
/* never be a valid byte).  Otherwise it returns the byte read. 	*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  if (!WaitForByte (Port))		/* If we didn't receive a byte	*/
    return (-1);			/* Return an error indicator	*/
  return (inportb (Port+RBR));		/* Otherwise return the byte	*/
}
/************************************************************************/



/************************************************************************/
int WaitForByte (int Port)
/************************************************************************/
/* This function waits at the serial port for a byte to be received.	*/
/* If nothing comes through after TM_OUT_MAX seconds, then it returns	*/
/* FALSE (0).  Otherwise it returns TRUE (1).				*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  time_t BeginTime, CurTime;
  BeginTime = time (&BeginTime);		/* Set up timer		*/
  while (!(inportb (Port+LSR) & RDR))		/* If there's nada and	*/
    if (difftime (time (&CurTime), BeginTime) > TM_OUT_MAX)  /* no time	*/
      return (FALSE);				/* inform caller of TO	*/
  return (TRUE);				/* Got one.		*/
}
/************************************************************************/



/************************************************************************/
void ClearPort (int Port)
/************************************************************************/
/* This function clears out any byte that is in the Receive Buffer of	*/
/* the port address indicated in the parameter.				*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  while (inportb (Port+LSR) & RDR)	/* While there is data to get	*/
    inportb (Port+RBR);			/* Get it and forget it		*/
}
/************************************************************************/
