/************************************************************************/
/*									*/
/*	INITFNS.C	"Initializer functions"				*/
/*									*/
/*   This file contains many of the functions used to initialize the	*/
/* CRYPTOCard using the Initializer.  It does not contain any of the 	*/
/* actual I/O functions, but it shows the method in which the data	*/
/* should be sent out.  The I/O functions are found in the files 	*/
/* "CARDFNS.C", "FIXFNS.C", and "PORT.C".				*/
/*									*/
/*   (C) - 1991 CRYPTOCard Corporation.  All rights reserved.		*/
/*									*/
/************************************************************************/

/************************************************************************/
/* Header files:							*/
/************************************************************************/
#include <dos.h>
#include <stdio.h>
#include <string.h>
#include <mem.h>
#include "intlzr.h"
#if defined VER32
#include "langv2.h"
#else
#include "langv1.h"
#endif


/************************************************************************/
/* External Variables:							*/
/************************************************************************/
extern int _Port = COM2;		/* Default port is COM1		*/
extern int _Lang = 0;			/* Language used in CRYPTOCard	*/
extern int _ErrorCode = 0;		/* Error return code from fns	*/



/************************************************************************/
/* Local Defines:							*/
/************************************************************************/
#define ENCRYPT		0		/* Used in DES algorithm	*/
#define DECRYPT		1		/* Ditto			*/


/************************************************************************/
int InitializeCard (CRYPTOCard * Card)
/************************************************************************/
/* This function calls another subset of functions that execute the	*/
/* necessary procedures needed to initialize a CRYPTOCard.  The data	*/
/* placed on the CRYPTOCard is taken from the structure pointed to in	*/
/* the parameters.  This structure contains all of the numeric values	*/
/* that a user entering data onto the CRYPTOCard's keypad would punch	*/
/* in.  For a more detailed description of the structure, see the files	*/
/* "INTLZR.C" and "INTLZR.H".  For a sample of how to enter data into	*/
/* the structure, see "MISC.C".						*/
/* This function returns FALSE (0) if, for any reason, the initializa-	*/
/* tion failed.  If it was successful, it returns TRUE (1).		*/
/* Dale Darling - May 8th, 1991.					*/
/************************************************************************/
{
  if (!LinkWithCard ())			/* If we can't link up then	*/
    return (_ErrorCode);		/* return code			*/

/*  printf ("Starting CRYPTOCard initialization.\n"); */

  SendByteToFixture (SOL|LED_INIT);	/* Start the INIT LED a-glowin'	*/
/*  printf ("Sending Options...\n"); */
  if (!SendOptions (Card)) {		/* Send the Option characters	*/
    SendByteToFixture (LED_NONE);	/* Shut off those LEDS		*/
    return (_ErrorCode);		/* If something went wrong...	*/
  }

/*  printf ("Sending Keys...\n"); */
  if (!SendKeys (Card))	{		/* Send the Keys, check for	*/
    SendByteToFixture (LED_NONE);	/* success. Shut off those LEDS	*/
    return (_ErrorCode);		/* If something went wrong...	*/
  }

/*  printf ("Sending UserID...\n"); */
  if (!SendUserID (Card)) {		/* This is becoming repetitive.	*/
    SendByteToFixture (LED_NONE);	/* Shut off those LEDS		*/
    return (_ErrorCode);		/* If something went wrong...	*/
  }

/*  printf ("Sending PIN...\n"); */
  if (!SendPIN (Card)) {
    SendByteToFixture (LED_NONE);	/* Shut off those LEDS		*/
    return (_ErrorCode);		/* If something went wrong...	*/
  }

  SendByteToFixture (LED_DONE);		/* Done LED pleases us muchly	*/
  SendByteToFixture (LED_NONE);		/* Turn 'em off.		*/

  return ((_ErrorCode=0));		/* If it got this far, then the	*/
}					/* card should be programmed.	*/
/************************************************************************/


/************************************************************************/
void SetPort (int Port)
/************************************************************************/
/* This function set the global variable _Port that is used by the	*/
/* functions in "CARDFNS.C" and "FIXFNS.C" to determine which port to	*/
/* communicate through.  It also initializes the port to the proper	*/
/* settings for communication with the Initializer (by calling the 	*/
/* function InitPort() found in "PORT.C").				*/
/* Dale Darling - May 8th, 1991.					*/
/************************************************************************/
{
  _Port = Port;
  InitPort (Port);
}
/************************************************************************/


/************************************************************************/
int LinkWithCard (void)
/************************************************************************/
/* This function attempts to link with the CRYPTOCard currently 	*/
/* inserted in the Initializer.  If it there is no card present, it 	*/
/* will wait until such a contigency has been met.  If the link was 	*/
/* successful, this function returns TRUE (1), otherwise it returns	*/
/* FALSE (0).								*/
/* Dale Darling - May 8th, 1991.					*/
/************************************************************************/
{
  int Counter;				/* Looping variable		*/
  int Linked = FALSE;			/* Default return value		*/
  char * BackDoor = "225371";		/* Back door reset sequence	*/
  char Buf[17];				/* Buffer for comparisons	*/

  if (!IsCardIn ()) {			/* If there's no Card in Fixtr.	*/
/*    printf ("Please insert CRYPTOCard into the Initializer.\n"); */
    while (!IsCardIn ());		/* Wait for ever		*/
  }					/* Card is now inserted		*/
  sleep (1);				/* Wait just a second		*/
  SendByteToFixture (SOL);		/* Engage Solenoid and Flash	*/
/*  printf ("Awaiting message from CRYPTOCard.\n"); */
  /* We don't know if the CRYPTOCard is on or off, so try both ways	*/
  for (Counter = 0; Counter < 2; Counter++) {
    SendByteToFixture (SOL|LED_RESET);	/* Light up the RESET LED	*/
    sleep (1);				/* Pause for but a moment	*/
    SendByteToFixture (SOL);		/* Shut RESET LED off		*/
    if (!GotThisBuf ("        ")) 	/* If we didn't get blanks...	*/
      continue;				/* it's not on (may try again)	*/
    SendDec (BackDoor, 6);		/* Send the backdoor sequence	*/
    strcpy (Buf, "        ");		/* 8 spaces			*/
    strcat (Buf, LanguageTable[0][LOCKED]);	/* Concat. "Locked"	*/
    if (GotThisBuf (Buf)) {		/* If it's on the screen...	*/
      Linked = TRUE;			/* Link was successful		*/
      break;				/* Break from loop		*/
    }					/* End of "if" block		*/
  }					/* End of "loop" block		*/
  if (!Linked) {				/* If not linked...		*/
/*    printf ("CRYPTOCard not responding.  Please remove.\n"); */
    _ErrorCode = CRYPT_NO_RSPNS;	/* Set the global error code	*/
    while (!(SendByteToFixture (LED_ERROR)) & 0x01);	/* Fire up the	*/
  }
/*  else				/* old ERROR LED.		*/
    printf ("CRYPTOCard communication established.\n"); */
  return (Linked);
}
/************************************************************************/




/************************************************************************/
int SendOptions (CRYPTOCard * Card)
/************************************************************************/
/* This function looks for the "Options?" message on the CRYPTOCard's	*/
/* display, and then sends the 3 options defined in the CRYPTOCard	*/
/* structure (pointed to in the parameter) across to the CRYPTOCard.	*/
/* If the "Options?" message is not found, it returns FALSE, otherwise	*/
/* it returns TRUE.							*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  int Counter;					/* That looper again	*/

  if (!SendByteToCard (ENT_KEY)) {	/* Send ENT: gets out of Locked	*/
/*    printf ("CRYPTOCard not responding.  Please remove.\n"); */
    _ErrorCode = CRYPT_NO_RSPNS;	/* Set the error indicator	*/
    while (SendByteToFixture (LED_ERROR) & 0x01)
      return (FALSE);			/* Flash an error and return	*/
  }					/* End the check		*/

  if (!GotThisBuf (LanguageTable[0][OPTIONS])) {/* Look for "Options?"	*/
    _ErrorCode = CRYPT_NO_OPTS;			/* Set error		*/
    return (FALSE);				/* message. If no go...	*/
  }
  for (Counter = 0; Counter < 3; Counter++)	/* Send the 3 options	*/
    if (!(SendOctal (Card->Options[Counter]))) {/* Just do it.		*/
      _ErrorCode = CRYPT_COM_ERR;		/* Uh oh...		*/
      return (FALSE);				/* Communication error.	*/
    }
  return (TRUE);				/* All's well...	*/
}
/************************************************************************/



/************************************************************************/
int SendKeys (CRYPTOCard * Card)
/************************************************************************/
/* This function sends the keys defined in the CRYPTOCard structure	*/
/* pointed to in the parameters across to the CRYPTOCard inserted in	*/
/* the Initializer.  It first calculates the number of keys by examin-	*/
/* ing the 3rd character in the 3rd Option (which just happens to be	*/
/* the number of keys), then sends them all across.  If there was a 	*/
/* communication error or the "Keyx?" messages couldn't be found, then	*/
/* this function returns FALSE (0).  Otherwise it returns TRUE (1).	*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  int NumKeys = Card->Options[2][2] - 0x30;	/* Calc. Number of keys	*/
  int Count, Count2;				/* There (s)he is again	*/
  int Decimal = (Card->Options[0][1] == '0') ? FALSE : TRUE;
  int Phone = (Card->Options[0][2] == '0') ? FALSE : TRUE;
  /* The above two lines calculate whether the display is in decimal	*/
  /* format and/or telephone format (see CRYPTOCard DES Challenge Token	*/
  /* Operations and Systems Guide, Revision 2.2, Page 5), as opposed to	*/
  /* the default mode - hexadecimal.					*/
  char Encrypt[20];				/* Holds encrypted rslt	*/

  _Lang = Card->Options[2][1] - 0x30;		/* The language used	*/
  for (Count = 0; Count < NumKeys; Count++) {
    SendByteToCard (ENT_KEY);			/* Move on to next Key	*/
    if (!GotThisBuf (LanguageTable[_Lang][Count])) { /* We got "Keyx?"	*/
      _ErrorCode = CRYPT_NO_KEYX;
      return (FALSE);				/* Nope, Sorry!		*/
    }
    GetEncryptedResult (Card->Keys[Count], Encrypt); /* Get DES result	*/
    if (Decimal)				/* If decimal display..	*/
      for (Count2 = 0; Count2 < 8; Count2++)	/* Go through encrypt.	*/
	if (Encrypt[Count2] >= 'A')		/* If it needs change	*/
	  Encrypt[Count2] -= 0x11;		/* Set it to a digit	*/
    if (Phone)					/* If phone display...	*/
       Encrypt[3] = '-';			/* Make 4th char "-".	*/
    for (Count2 = 0; Count2 < 8; Count2++)	/* Send 8 octal digits	*/
      SendOctal (Card->Keys[Count][Count2]);	/* Send this octal num	*/
    SendByteToCard (ENT_KEY);			/* Send an ENTer next	*/
    /* CRYPTOCard should do an encryption on the key using 00000000,	*/
    /* and send back the result.  We'll compare the CRYPTOCard's to	*/
    /* ours to make sure it is working fine.				*/
    if (!(GotThisBuf (Encrypt))) {		/* Didn't get it back..	*/
/*      printf ("\nEncrypted results do not match.\n"); */
      while (SendByteToFixture (LED_ERROR) & 0x01); /* Lighten up ERROR	*/
      _ErrorCode = CRYPT_BAD_ENCRPT;		/* Set code		*/
      return (FALSE);				/* Indicate error	*/
    }
  }						/* End the NumKeys loop	*/
  return (TRUE);				/* We are happy.	*/
}
/************************************************************************/




/************************************************************************/
void GetEncryptedResult (oct * Key, char * Result)
/************************************************************************/
/* This function will encrypt the 8 octal digits pointed to in the Key	*/
/* parameter using 00000000 and a DES algorithm, and place the result	*/
/* (which is in hexadecimal format) into the buffer pointed to in the 	*/
/* parameters.  The buffer contains the ASCII version of each hex	*/
/* digit.  The DES algorithm is linked in from the file "DES.OBJ", 	*/
/* which came from a shareware package.					*/
/* Dale Darling and Steven Seal - May 9th, 1991.			*/
/************************************************************************/
{
  char BinKey[9];			/* The ASCII version of the key	*/
  char Zeros[9];			/* The encryption uses zeros	*/
  char Encrypt[17];			/* Copy of the final result	*/
  char Temp[3];				/* Temporary holder of hex val	*/
  unsigned Function = ENCRYPT;		/* Used in DES call		*/
  int Count;				/* Iteration variable		*/

  OctToBin (BinKey, Key, 8);		/* Convert the Octal key to Bin	*/
  setmem (Zeros, 8, 0x00);		/* Fill with zeros		*/
  Result[0] = NULL;			/* Clear the string		*/

/*  printf("\n");
  for (Count = 0; Count < 8; Count++) {
    printf ("%d ", BinKey[Count]);
  }

  printf ("\nAbove is the binary version.\n");
*/

  DES (Encrypt, Zeros, BinKey, Function);	/* Do the DES algorithm	*/

  for (Count = 0; Count < 4; Count++) {
    sprintf (Temp, "%02X", Encrypt[Count] & 0x00ff);
    strcat (Result, Temp);
  }

  Result[8] = NULL;
/*  printf ("%s", Result);
  printf ("\nAbove is the encrypted result's hex version.\n");
*/

}
/************************************************************************/



/************************************************************************/
int SendUserID (CRYPTOCard * Card)
/************************************************************************/
/* This function first determines whether it needs to send the User ID	*/
/* or not, according to the Options.  If not, it returns a TRUE to 	*/
/* indicate that no errors occurred (even though it didn't read the ID)	*/
/* It then gets an ASCII version of the User's ID to be used when	*/
/* checking that it was entered correctly.  It then sends the octal 	*/
/* version of the ID to the CRYPTOCard and tests the echo to make sure	*/
/* that all is well.  If something has gone amiss, the function returns	*/
/* FALSE, otherwise we get a gleeful TRUE.				*/
/* Note that the User ID should be 8 bytes long.  If it is only 7, a	*/
/* white space character should be appended.				*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  char UserID[9];			/* Buffer to hold the ASCII ID	*/
  int Counter;				/* My friend.			*/

  if (Card->Options[1][0] == '0')	/* User ID not required!	*/
    return (TRUE);			/* Get on outa here.		*/

  SendByteToCard (ENT_KEY);		/* Move on to User ID.		*/
  if (!(GotThisBuf (LanguageTable[_Lang][USER_ID]) )) {
    _ErrorCode = CRYPT_NO_ID;		/* Set the code			*/
    return (FALSE);			/* "User ID" not displayed.	*/
  }

  OctToASCII (UserID, Card->UserID, 8);	/* Get an ASCII version.	*/
  for (Counter = 0; Counter < 8; Counter++) {
    SendOctal (Card->UserID[Counter]);	/* Send each octal byte to Card	*/
/*    GetByte (_Port);		*/	/* Get byte num and ignore it	*/
  }
  SendByteToCard (ENT_KEY);		/* Complete the sequence	*/

  if (!(GotThisBuf (UserID) )) {	/* ID doesn't match!		*/
    _ErrorCode = CRYPT_BAD_ID;		/* set value			*/
    return (FALSE);			/* Inform the caller.		*/
  }

  return (TRUE);			/* All is fine and dandy.	*/
}
/************************************************************************/




/************************************************************************/
int SendPIN (CRYPTOCard * Card)
/************************************************************************/
/* This function sends the User's PIN across to the CRYPTOCard.  It	*/
/* then sends it again to satisfy the CRYPTOCard's need for		*/
/* verification.							*/
/* Dale Darling - May 9th, 1991.					*/
/************************************************************************/
{
  SendByteToCard (ENT_KEY);		/* Move CRYPTOCard to next step	*/

  if (!(GotThisBuf (LanguageTable[_Lang][NEW_PIN]) )) {
    _ErrorCode = CRYPT_NO_NEWPIN;
    return (FALSE);
  }

  SendDec (Card->UserPIN, 8);		/* Send this PIN across now	*/
  SendByteToCard (ENT_KEY);		/* Finish with ENTer		*/

  if (!(GotThisBuf (LanguageTable[_Lang][VERIFY]) )) {
    _ErrorCode = CRYPT_NO_VER;
    return (FALSE);
  }

  SendDec (Card->UserPIN, 8);		/* Send it yet again		*/
  SendByteToCard (ENT_KEY);		/* Close it off			*/

  if (!(GotThisBuf (LanguageTable[_Lang][CARD_OK]) )) {
    _ErrorCode = CRYPT_NO_OK;
    return (FALSE);			/* Oops				*/
  }

  SendByteToCard (ENT_KEY);		/* Move on to PIN? message	*/

  if (!(GotThisBuf (LanguageTable[_Lang][PIN]) )) {
    _ErrorCode = CRYPT_NO_PIN;
    return (FALSE);
  }

  return (TRUE);			/* Card done successfully	*/
} 					/* So long!			*/
/************************************************************************/
