                PAGE    60,132

                TITLE   DES (Data Encryption Standard) Algorithm

                SUBTTL  (c) 1988-89 MicrOhio

;       Implementation of the Data Encryption Standard as published by
;       the National Bureau of Standards, "Data Encryption Standard",
;       FIPS (Federal Information Processing Standard) Publication 46.
;
;       On a 4.77 MHz Intel 8088 (not V20) based PC, over 2500 bits
;       per second may be processed (using the same key for each 64-bit
;       block).
;
;       C callable functions contained herein:
;
;               DES ENCRYPT DECRYPT SETKEY
;
;       All are designed to be called from a C language program.
;       They are also callable from assembler language programs
;       in an efficient manner. In SMALL and MEDIUM model, it is
;       assumed that the DS register addresses the caller's data
;       segment. COMPACT and LARGE models assume nothing about the
;       DS register. Registers DS, ES, BP, DI, and SI are preserved
;       by these functions. If your compiler demands that assembly
;       language functions not modify other registers, edit the
;       PROLOG and EPILOG macros as needed. In strict accordance
;       with C calling conventions, the calling function removes
;       the passed arguements from the stack.
;
;       Model           Characteristics
;       -----           ---------------
;       SMALL           NEAR code, NEAR data
;       MEDIUM          FAR code, NEAR data
;       COMPACT         NEAR code, FAR data
;       LARGE           FAR code, FAR data
;
;       Copyright (c) 1988-89 by MicrOhio  All rights reserved

                .SALL                           ; remove for full listing

                NAME    DES

ENCRYPT         EQU     0                       ; function code to encrypt text
DECRYPT         EQU     1                       ; function code to decrypt text

SMALL           EQU     0                       ; select one model
MEDIUM          EQU     0
COMPACT         EQU     0
LARGE           EQU     1

                PAGE

                IF2
                IF      (SMALL + MEDIUM + COMPACT + LARGE) NE 1
                %OUT    Wrong model specified
                ENDIF
                IF      SMALL
                %OUT    SMALL model
                ENDIF
                IF      MEDIUM
                %OUT    MEDIUM model
                ENDIF
                IF      COMPACT
                %OUT    COMPACT model
                ENDIF
                IF      LARGE
                %OUT    LARGE model
                ENDIF
                ENDIF

;       Save C language reserved registers,
;       establish frame pointer, reset direction flag

PROLOG          MACRO
                PUSH    BP
                MOV     BP,SP
                PUSH    SI
                PUSH    DI
                PUSH    DS
                PUSH    ES
                CLD
                ENDM

;       Restore C language reserved registers

EPILOG          MACRO
                POP     ES
                POP     DS
                POP     DI
                POP     SI
                POP     BP
                ENDM

;       Expand 64 bits at DS:SI to 64 bytes at ES:DI

EXPAND          MACRO
                XOR     DL,DL
                REPT    8
                LODSB
                MOV     AH,AL
                REPT    8
                MOV     AL,DL
                ROL     AX,1
                STOSB
                ENDM
                ENDM
                ENDM

;       Pack 64 bytes at DS:SI to 64 bits at ES:DI

PACK            MACRO
                REPT    8
                REPT    8
                SHL     AL,1
                OR      AL,[SI]
                INC     SI
                ENDM
                STOSB
                ENDM
                ENDM

;       Scatter bytes at DS:SI to bytes at ES:DI

PERMUTE         MACRO   DISP
                IRP     D,<DISP>
                MOV     AL,[SI + D]
                STOSB
                ENDM
                ENDM

                PAGE

$DES_DATA       SEGMENT WORD PUBLIC 'DATA'

SUBKEY          DB      (48 * 16) DUP (0)       ; array of 48 sub-keys
CD              DB      56 DUP (0)              ; concatenated C and D
TEMP64A         DB      64 DUP (0)              ; gp buffer (1 byte per bit)
TEMP64B         DB      64 DUP (0)              ; gp buffer (1 byte per bit)
TEMP32          DB      32 DUP (0)              ; gp buffer (1 byte per bit)
TEMP48          DB      48 DUP (0)              ; gp buffer (1 byte per bit)
PTR1            DW      0                       ; used by CIPHER/DECIPHER
PTR2            DW      0                       ; used by CIPHER/DECIPHER
ROTC            DB      1,1,2,2,2,2,2,2         ; rotate counts for SETKEY
                DB      1,2,2,2,2,2,2,1         ; (initialized data)

.XLIST

SBOX    DB      224,0,64,240,208,112,16,64,32,224,240,32,176,208,128,16
        DB      48,160,160,96,96,192,192,176,80,144,144,80,0,48,112,128
        DB      64,240,16,192,224,128,128,32,208,64,96,144,32,16,176,112
        DB      240,80,192,176,144,48,112,224,48,160,160,0,80,96,0,208
        DB      240,48,16,208,128,64,224,112,96,240,176,32,48,128,64,224
        DB      144,192,112,0,32,16,208,160,192,96,0,144,80,176,160,80
        DB      0,208,224,128,112,160,176,16,160,48,64,240,208,64,16,32
        DB      80,176,128,96,192,112,96,192,144,0,48,80,32,224,240,144
        DB      160,208,0,112,144,0,224,144,96,48,48,64,240,96,80,160
        DB      16,32,208,128,192,80,112,224,176,192,64,176,32,240,128,16
        DB      208,16,96,160,64,208,144,0,128,96,240,144,48,128,0,112
        DB      176,64,16,240,32,224,192,48,80,176,160,80,224,32,112,192
        DB      112,208,208,128,224,176,48,80,0,96,96,240,144,0,160,48
        DB      16,64,32,112,128,32,80,192,176,16,192,160,64,224,240,144
        DB      160,48,96,240,144,0,0,96,192,160,176,16,112,208,208,128
        DB      240,144,16,64,48,80,224,176,80,192,32,112,128,32,64,224
        DB      32,224,192,176,64,32,16,192,112,64,160,112,176,208,96,16
        DB      128,80,80,0,48,240,240,160,208,48,0,144,224,128,144,96
        DB      64,176,32,128,16,192,176,112,160,16,208,224,112,32,128,208
        DB      240,96,144,240,192,0,80,144,96,160,48,64,0,80,224,48
        DB      192,160,16,240,160,64,240,32,144,112,32,192,96,144,128,80
        DB      0,96,208,16,48,208,64,224,224,0,112,176,80,48,176,128
        DB      144,64,224,48,240,32,80,192,32,144,128,80,192,240,48,160
        DB      112,176,0,224,64,16,160,112,16,96,208,0,176,128,96,208
        DB      64,208,176,0,32,176,224,112,240,64,0,144,128,16,208,160
        DB      48,224,192,48,144,80,112,192,80,32,160,240,96,128,16,96
        DB      16,96,64,176,176,208,208,128,192,16,48,64,112,160,224,112
        DB      160,144,240,80,96,0,128,240,0,224,80,32,144,48,32,192
        DB      208,16,32,240,128,208,64,128,96,160,240,48,176,112,16,64
        DB      160,192,144,80,48,96,224,176,80,0,0,224,192,144,112,32
        DB      112,32,176,16,64,224,16,112,144,64,192,160,224,128,32,208
        DB      0,240,96,192,160,144,208,0,240,48,48,80,80,96,128,176
        DB      155,155,141,114,145,154,141,104

.LIST

$DES_DATA       ENDS

                PAGE

                IF      SMALL OR COMPACT
_TEXT           SEGMENT WORD PUBLIC 'CODE'
                ASSUME  CS:_TEXT
                ELSE
$DES_TEXT       SEGMENT WORD PUBLIC 'CODE'
                ASSUME  CS:$DES_TEXT
                ENDIF

                ASSUME  DS:$DES_DATA
                ASSUME  ES:$DES_DATA

                PUBLIC  _DES
                PUBLIC  _ENCRYPT
                PUBLIC  _DECRYPT
                PUBLIC  _SETKEY

                PAGE

;       Entry point for a single block encrypt or decrypt operation
;
;       Usage:  des(dest, srce, key, func)
;
;       Where:  dest (char *) = encrypted/decrypted data placed here
;               srce (char *) = pointer to 64 bits of data
;               key (char *) = pointer to 64 bits of key data
;               func (int) = 0 to encrypt, 1 to decrypt
;
;       Return: void
;
;       Note:   Arguements pushed right to left (C style).
;               Registers DS, ES, BP, DI and SI preserved.
;               Key parity (should be odd) is not checked.

                IF      SMALL
D_STACK_FRAME   STRUC
D_REGISTER_BP   DW      ?
D_RETURN        DW      ?
D_DEST          DW      ?
D_SRCE          DW      ?
D_KEY           DW      ?
D_FUNC          DW      ?
D_STACK_FRAME   ENDS
                ENDIF

                IF      MEDIUM
D_STACK_FRAME   STRUC
D_REGISTER_BP   DW      ?
D_RETURN        DD      ?
D_DEST          DW      ?
D_SRCE          DW      ?
D_KEY           DW      ?
D_FUNC          DW      ?
D_STACK_FRAME   ENDS
                ENDIF

                IF      COMPACT
D_STACK_FRAME   STRUC
D_REGISTER_BP   DW      ?
D_RETURN        DW      ?
D_DEST          DD      ?
D_SRCE          DD      ?
D_KEY           DD      ?
D_FUNC          DW      ?
D_STACK_FRAME   ENDS
                ENDIF

                IF      LARGE
D_STACK_FRAME   STRUC
D_REGISTER_BP   DW      ?
D_RETURN        DD      ?
D_DEST          DD      ?
D_SRCE          DD      ?
D_KEY           DD      ?
D_FUNC          DW      ?
D_STACK_FRAME   ENDS
                ENDIF

                IF      SMALL OR COMPACT
_DES            PROC    NEAR
                ELSE
_DES            PROC    FAR
                ENDIF
                IF      SMALL OR MEDIUM
                PROLOG
                MOV     SI,[BP].D_KEY           ; ptr to key
                PUSH    DS
                CALL    $SETKEY                 ; generate the 16 subkeys
                POP     DS
                MOV     AX,DS                   ; establish addressability
                MOV     ES,AX
                MOV     CX,[BP].D_FUNC          ; get the function code
                MOV     SI,[BP].D_SRCE          ; point to source
                MOV     DI,[BP].D_DEST          ; point to destination
                CALL    $DES                    ; perform the requested op
                EPILOG
                RET
                ELSE
                PROLOG
                LDS     SI,[BP].D_KEY           ; pass ptr to real key
                CALL    $SETKEY                 ; generate the 16 subkeys
                MOV     CX,[BP].D_FUNC          ; get the function code
                LDS     SI,[BP].D_SRCE          ; point to source
                LES     DI,[BP].D_DEST          ; point to destination
                CALL    $DES                    ; perform the requested op
                EPILOG
                RET
                ENDIF
_DES            ENDP

                PAGE

;       Entry point for a multiple block encrypt operation
;
;       Usage:  encrypt(dest, srce)
;
;       Where:  dest (char *) = encrypted data placed here
;               srce (char *) = pointer to 64 bits of plaintext data
;
;       Return: void
;
;       Note:   Arguements pushed right to left (C style).
;               Registers DS, ES, BP, DI and SI preserved.

                IF      SMALL
E_STACK_FRAME   STRUC
E_REGISTER_BP   DW      ?
E_RETURN        DW      ?
E_DEST          DW      ?
E_SRCE          DW      ?
E_STACK_FRAME   ENDS
                ENDIF

                IF      MEDIUM
E_STACK_FRAME   STRUC
E_REGISTER_BP   DW      ?
E_RETURN        DD      ?
E_DEST          DW      ?
E_SRCE          DW      ?
E_STACK_FRAME   ENDS
                ENDIF

                IF      COMPACT
E_STACK_FRAME   STRUC
E_REGISTER_BP   DW      ?
E_RETURN        DW      ?
E_DEST          DD      ?
E_SRCE          DD      ?
E_STACK_FRAME   ENDS
                ENDIF

                IF      LARGE
E_STACK_FRAME   STRUC
E_REGISTER_BP   DW      ?
E_RETURN        DD      ?
E_DEST          DD      ?
E_SRCE          DD      ?
E_STACK_FRAME   ENDS
                ENDIF

                IF      SMALL OR COMPACT
_ENCRYPT        PROC    NEAR
                ELSE
_ENCRYPT        PROC    FAR
                ENDIF
                IF      SMALL OR MEDIUM
                PROLOG
                MOV     CX,ENCRYPT              ; set ENCRYPT function
                MOV     AX,DS                   ; establish addressability
                MOV     ES,AX
                MOV     SI,[BP].E_SRCE          ; point to source
                MOV     DI,[BP].E_DEST          ; point to destination
                CALL    $DES                    ; perform the requested op
                EPILOG
                RET
                ELSE
                PROLOG
                MOV     CX,ENCRYPT              ; set ENCRYPT function
                LDS     SI,[BP].E_SRCE          ; point to source
                LES     DI,[BP].E_DEST          ; point to destination
                CALL    $DES                    ; perform the requested op
                EPILOG
                RET
                ENDIF
_ENCRYPT        ENDP

                PAGE

;       Entry point for a multiple block decrypt operation
;
;       Usage:  decrypt(dest, srce)
;
;       Where:  dest (char *) = decrypted data placed here
;               srce (char *) = pointer to 64 bits of plaintext data
;
;       Return: void
;
;       Note:   Arguements pushed right to left (C style).
;               Registers DS, ES, BP, DI and SI preserved.

                IF      SMALL OR COMPACT
_DECRYPT        PROC    NEAR
                ELSE
_DECRYPT        PROC    FAR
                ENDIF
                IF      SMALL OR MEDIUM
                PROLOG
                MOV     AX,DS                   ; establish addressability
                MOV     ES,AX
                MOV     CX,DECRYPT              ; set DECRYPT function
                MOV     SI,[BP].E_SRCE          ; point to source
                MOV     DI,[BP].E_DEST          ; point to destination
                CALL    $DES                    ; perform the requested op
                EPILOG
                RET
                ELSE
                PROLOG
                MOV     CX,DECRYPT              ; set DECRYPT function
                LDS     SI,[BP].E_SRCE          ; point to source
                LES     DI,[BP].E_DEST          ; point to destination
                CALL    $DES                    ; perform the requested op
                EPILOG
                RET
                ENDIF
_DECRYPT        ENDP

                PAGE

;       Generate iterated subkeys using the key schedule function
;
;       Usage:  setkey(key)
;
;       Where:  key (char *) = pointer to 64 bits of key data
;
;       Return: void
;
;       Note:   Key parity (should be odd) is not checked.
;               Registers DS, ES, BP, DI and SI preserved.

                IF      SMALL
S_STACK_FRAME   STRUC
S_REGISTER_BP   DW      ?
S_RETURN        DW      ?
S_KEY           DW      ?
S_STACK_FRAME   ENDS
                ENDIF

                IF      MEDIUM
S_STACK_FRAME   STRUC
S_REGISTER_BP   DW      ?
S_RETURN        DD      ?
S_KEY           DW      ?
S_STACK_FRAME   ENDS
                ENDIF

                IF      COMPACT
S_STACK_FRAME   STRUC
S_REGISTER_BP   DW      ?
S_RETURN        DW      ?
S_KEY           DD      ?
S_STACK_FRAME   ENDS
                ENDIF

                IF      LARGE
S_STACK_FRAME   STRUC
S_REGISTER_BP   DW      ?
S_RETURN        DD      ?
S_KEY           DD      ?
S_STACK_FRAME   ENDS
                ENDIF

                IF      SMALL OR COMPACT
_SETKEY         PROC    NEAR
                ELSE
_SETKEY         PROC    FAR
                ENDIF
                IF      SMALL OR MEDIUM
                PROLOG
                MOV     SI,[BP].S_KEY           ; point to passed key
                CALL    $SETKEY
                EPILOG
                RET
                ELSE
                PROLOG
                LDS     SI,[BP].S_KEY           ; point to passed key
                CALL    $SETKEY
                EPILOG
                RET
                ENDIF
_SETKEY         ENDP

                PAGE

;       Generate all iterated subkeys using the key schedule function
;
;       Where:  DS:SI = pointer to 64 bits of key data
;
;       Return: void
;
;       Note:   This function is NEAR.
;               The direction flag is known to be cleared.
;               Key parity (should be odd) is not checked.

$SETKEY         PROC    NEAR
                MOV     AX,SEG $DES_DATA        ; establish addressability
                MOV     ES,AX
                MOV     DI,OFFSET TEMP64A
                EXPAND                          ; expand 64 key bits to bytes

;       Generate CD using the PC1 permutation table

                MOV     AX,ES                   ; establish addressability
                MOV     DS,AX
                MOV     DI,OFFSET CD
                MOV     SI,OFFSET TEMP64A
                PERMUTE <56,48,40,32,24,16,8,0,57,49,41,33,25,17>
                PERMUTE <9,1,58,50,42,34,26,18,10,2,59,51,43,35>
                PERMUTE <62,54,46,38,30,22,14,6,61,53,45,37,29,21>
                PERMUTE <13,5,60,52,44,36,28,20,12,4,27,19,11,3>

;       Generate the 16 subkeys by performing rotates and permutations

                MOV     CX,16                   ; init loop counter
                MOV     BX,OFFSET ROTC          ; init ptr to rotate count
                MOV     DX,OFFSET SUBKEY        ; init ptr to subkey array

$SET1:          MOV     AH,[BX]                 ; get correct rotate count

$SET2:          MOV     SI,OFFSET CD            ; rotate cd by rotc[loop]
                MOV     DI,SI                   ; array start address to DI
                LODSB                           ; save first byte of array
                REPT    13                      ; move 27 chars (13 words + 1)
                MOVSW
                ENDM
                MOVSB
                MOV     [DI],AL                 ; first byte is now the last
                DEC     AH
                JNZ     $SET2                   ; loop till all done..
                MOV     AH,[BX]                 ; get correct rotate count

$SET3:          MOV     SI,OFFSET CD + 28       ; rotate cd + 28 by rotc[loop]
                MOV     DI,SI                   ; array start address to DI
                LODSB                           ; save first byte of array
                REPT    13                      ; move 27 chars (13 words + 1)
                MOVSW
                ENDM
                MOVSB
                MOV     [DI],AL                 ; first byte is now the last
                DEC     AH
                JNZ     $SET3                   ; loop till all done..

                MOV     DI,DX                   ; permute cd to skey[loop]
                MOV     SI,OFFSET CD
                PERMUTE <13,16,10,23,0,4,2,27,14,5,20,9>
                PERMUTE <22,18,11,3,25,7,15,6,26,19,12,1>
                PERMUTE <40,51,30,36,46,54,29,39,50,44,32,47>
                PERMUTE <43,48,38,55,33,52,45,41,49,35,28,31>

                ADD     DX,48                   ; update ptr to subkey
                INC     BX                      ; update rotate count ptr
                LOOP    $SET4                   ; do all 16 loops...
                RET

$SET4:          JMP     $SET1
$SETKEY         ENDP

                PAGE

;       Entry point for a single block encrypt or decrypt operation
;
;       Where:  ES:DI = encrypted/decrypted data placed here
;               DS:SI = pointer to 64 bits of data
;               CX = 0 to encrypt, 1 to decrypt
;
;       Return: void
;
;       Note:   This function is NEAR.
;               The direction flag is known to be cleared.
;               $SETKEY must have been called prior.

$DES            PROC    NEAR
                PUSH    ES                      ; save destination address
                PUSH    DI
                MOV     AX,SEG $DES_DATA        ; establish addressability
                MOV     ES,AX
                MOV     DI,OFFSET TEMP64A
                EXPAND                          ; expand 64 data bits to bytes

;       Perform the initial permutation of the data

                MOV     AX,ES                   ; establish addressability
                MOV     DS,AX
                MOV     DI,OFFSET TEMP64B
                MOV     SI,OFFSET TEMP64A
                PERMUTE <57,49,41,33,25,17,9,1,59,51,43,35,27,19,11,3>
                PERMUTE <61,53,45,37,29,21,13,5,63,55,47,39,31,23,15,7>
                PERMUTE <56,48,40,32,24,16,8,0,58,50,42,34,26,18,10,2>
                PERMUTE <60,52,44,36,28,20,12,4,62,54,46,38,30,22,14,6>

;       Generate the proper R and L based on the requested process

                JCXZ    $DES1                   ; go if encrypting...

                MOV     DI,OFFSET TEMP64B + 32
                MOV     SI,OFFSET TEMP64B
                CALL    DECIPHER
                JMP     SHORT $DES2

$DES1:          MOV     DI,OFFSET TEMP64B
                MOV     SI,OFFSET TEMP64B + 32
                CALL    CIPHER

;       Concatenate R and L to form the pre-output block

$DES2:          MOV     DI,OFFSET TEMP64A
                MOV     SI,OFFSET TEMP64B + 32
                REPT    16
                MOVSW
                ENDM
                MOV     SI,OFFSET TEMP64B
                REPT    16
                MOVSW
                ENDM

;       Do the inverse initial permutation to form the output block

                MOV     DI,OFFSET TEMP64B
                MOV     SI,OFFSET TEMP64A
                PERMUTE <39,7,47,15,55,23,63,31,38,6,46,14,54,22,62,30>
                PERMUTE <37,5,45,13,53,21,61,29,36,4,44,12,52,20,60,28>
                PERMUTE <35,3,43,11,51,19,59,27,34,2,42,10,50,18,58,26>
                PERMUTE <33,1,41,9,49,17,57,25,32,0,40,8,48,16,56,24>

;       Pack the 64 bytes back into 64 bits and return them to the caller

                POP     DI                      ; restore destination address
                POP     ES
                MOV     SI,OFFSET TEMP64B
                PACK
                RET
$DES            ENDP

                PAGE

;       Encrypt an expanded 64 byte block.
;
;       Where:  DI = left half of the Initial Permuted Data
;               SI = right half of the Initial Permuted Data
;               DS,ES = $DES_DATA segment
;
;       Return: void
;
;       Note:   This function is NEAR.
;               The direction flag is known to be cleared.
;               $SETKEY must have been called prior.

CIPHER          PROC    NEAR
                MOV     PTR1,DI
                MOV     PTR2,SI
                MOV     DX,OFFSET SUBKEY
                MOV     CX,16                   ; doing 16 loops

;       Permute 32 bytes into 48 bytes using the E table.
;       XOR the 48 byte result with the proper 48 byte subkey.
;       Run the result through the SELECT function to obtain
;       a 32 byte result. Permute that and store the result
;       in the first 32 bytes of TEMP48. XOR TEMP48 and LEFT,
;       yielding new RIGHT. Copy old RIGHT over LEFT. Repeat
;       sixteen times.

CIP1:           MOV     DI,OFFSET TEMP48        ; permute right to temp48
                MOV     SI,PTR2
                MOV     AL,[SI + 31]            ; offset = 31
                STOSB
                MOVSW                           ; offset = 0 - 4
                MOVSW
                MOVSB
                SUB     SI,2                    ; offset = 3 - 8
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 7 - 12
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 11 - 16
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 15 - 20
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 19 - 24
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 23 - 28
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 27 - 31
                MOVSW
                MOVSW
                MOVSB
                SUB     SI,32                   ; offset = 0
                MOVSB

                MOV     DI,OFFSET TEMP48        ; xor temp48 and subkey[iter]
                MOV     SI,DX
                REPT    24
                LODSW
                XOR     AX,[DI]
                STOSW
                ENDM

                MOV     DI,OFFSET TEMP32        ; select temp32 from temp48
                MOV     SI,OFFSET TEMP48
                CALL    SELECT

                MOV     DI,OFFSET TEMP48        ; permute temp32 to temp48
                MOV     SI,OFFSET TEMP32
                PERMUTE <15,6,19,20,28,11,27,16,0,14,22,25,4,17,30,9>
                PERMUTE <1,7,23,13,31,26,2,8,18,12,29,5,21,10,3,24>

                MOV     DI,PTR1                 ; xor left and temp48
                MOV     SI,OFFSET TEMP48
                REPT    16
                LODSW
                XOR     AX,[DI]
                STOSW
                ENDM

                MOV     AX,PTR2                 ; swap LEFT and RIGHT
                XCHG    AX,PTR1
                MOV     PTR2,AX

                ADD     DX,48                   ; point to next subkey
                LOOP    CIP2
                RET                             ; exit after 16 iterations...

CIP2:           JMP     CIP1
CIPHER          ENDP

                PAGE

;       Decrypt an expanded 64 byte block.
;
;       Where:  DI = left half of the Initial Permuted Data
;               SI = right half of the Initial Permuted Data
;               DS,ES = $DES_DATA segment
;
;       Return: void
;
;       Note:   This function is NEAR.
;               The direction flag is known to be cleared.
;               $SETKEY must have been called prior.

DECIPHER        PROC    NEAR
                MOV     PTR1,DI
                MOV     PTR2,SI
                MOV     DX,OFFSET SUBKEY
                ADD     DX,48*15                ; point to last subkey in array
                MOV     CX,16                   ; doing 16 loops

;       Permute 32 bytes into 48 bytes using the E table.
;       XOR the 48 byte result with the proper 48 byte subkey.
;       Run the result through the SELECT function to obtain
;       a 32 byte result. Permute that and store the result
;       in the first 32 bytes of TEMP48. XOR TEMP48 and RIGHT,
;       yielding new LEFT. Copy old LEFT over RIGHT. Repeat
;       sixteen times.

DEC1:           MOV     DI,OFFSET TEMP48        ; permute left to temp48
                MOV     SI,PTR1
                MOV     AL,[SI + 31]            ; offset = 31
                STOSB
                MOVSW                           ; offset = 0 - 4
                MOVSW
                MOVSB
                SUB     SI,2                    ; offset = 3 - 8
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 7 - 12
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 11 - 16
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 15 - 20
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 19 - 24
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 23 - 28
                MOVSW
                MOVSW
                MOVSW
                SUB     SI,2                    ; offset = 27 - 31
                MOVSW
                MOVSW
                MOVSB
                SUB     SI,32                   ; offset = 0
                MOVSB

                MOV     DI,OFFSET TEMP48        ; xor temp48 and subkey[iter]
                MOV     SI,DX
                REPT    24
                LODSW
                XOR     AX,[DI]
                STOSW
                ENDM

                MOV     DI,OFFSET TEMP32        ; select temp32 from temp48
                MOV     SI,OFFSET TEMP48
                CALL    SELECT

                MOV     DI,OFFSET TEMP48        ; permute temp32 to temp48
                MOV     SI,OFFSET TEMP32
                PERMUTE <15,6,19,20,28,11,27,16,0,14,22,25,4,17,30,9>
                PERMUTE <1,7,23,13,31,26,2,8,18,12,29,5,21,10,3,24>

                MOV     DI,PTR2                 ; xor right and temp48
                MOV     SI,OFFSET TEMP48
                REPT    16
                LODSW
                XOR     AX,[DI]
                STOSW
                ENDM

                MOV     AX,PTR2                 ; swap LEFT and RIGHT
                XCHG    AX,PTR1
                MOV     PTR2,AX

                SUB     DX,48                   ; point to next subkey
                LOOP    DEC2
                RET                             ; exit after 16 iterations...

DEC2:           JMP     DEC1
DECIPHER        ENDP

                PAGE

;       Perform the SELECT function
;
;       Where:  DI = pointer to destination
;               SI = pointer to source data
;               DS,ES = $DES_DATA segment
;
;       The selection process transforms a 48 byte array into
;       a 32 byte array. Source data is divided into 8 fields
;       of 6 bytes each. Traditionally, the first and sixth byte
;       of each field form a row value, while the second through
;       fifth bytes combine to form a column value. The row and
;       column are then used to select a value out of a table
;       (table number used depends on field number) and then this
;       value is expanded into four bytes and replaces the original
;       six byte field value.
;
;       This implementation re-arranges the S-Boxes to allow for
;       a simpler means of calculating the index into the tables.
;       The values in the S-Boxes have been left-shifted to exploit
;       an 8088's ability to easily rotate the MSB into the LSB.
;
;       Return: void

SELECT          PROC    NEAR
                MOV     BX,OFFSET SBOX  ; establish ptr to the SBOX
                REPT    8

;       Convert next six source data bits to an index (0 - 63)

                LODSB
                REPT    5
                SHL     AL,1
                OR      AL,[SI]
                INC     SI
                ENDM
                XLAT                    ; (AL) = SBOX value (shifted left 4)

;       Unpack next 4 bits into destination

                MOV     AH,AL
                REPT    4
                XOR     AL,AL           ; zero register
                ROL     AX,1            ; high-bit into low-bit
                STOSB                   ; store the result, 1 or 0
                ENDM
                ADD     BX,64           ; point to next section of SBOX
                ENDM
                RET
SELECT          ENDP

                IF      SMALL OR COMPACT
_TEXT           ENDS
                ELSE
$DES_TEXT       ENDS
                ENDIF

                END
