unit oenav;
{------------------------------------------------------------------------------
 OENavigator and OEExtHstmt components.
 These MUST be used with the ODBCExpress ODBC component set.

 Author: Greg Carter, with a lot of help from Borland and OE.
 Organization: CRYPTOCard Corporation
 Contact: gregc@cryptocard.com, www.cryptocard.com/pascal.html

 Components:
 OENavigator is similar to Delphi's built in DBNavigator.  Major differences
 include
 Uses OEExtHstmt as a datasource instead of TDatasource.
 OnReturnToBrowse: Use two OEExtHstmt, one to browse data, another for inserting
                   data. When the Insert button is pressed the OnInsert method
                   of the Browsing Hstmt is called.  Use the OnInsert and After
                   Insert events of the Hstmt to get user input, execute INSERT
                   statemets, and position the second Hstmt at the inserted
                   record.  Then when the user presses any of the move buttons(
                   next, first, prior, last) the OnReturnToBrowse is called.  You
                   can then set the Datasource back to the Browse Hstmt, so it
                   appears as though you did a positioned Insert.

 OEExtHstmt:
 Adds a number of events to the THstmt to make it act like a Delphi TTable.
 Events include AfterFirst, AfterPrior, AfterNext, AfterLast, AfterCancel, After
 Delete, AfterPost, AfterRefresh, BeforeCancel, BeforeDelete, BeforePost,
 BeforeRefresh, OnInsert, AfterInsert and OnError.

 OnInsert and AfterInsert were added to allow easy integration with OENav.
 Note no data is inserted when OnInsert is called.  This is only an event
 and it is upto the programmer to write the necessary code to insert a record.

 OnError will get triggered any time an ODBC error occures during the Post,
 Cancel, Refresh or Delete events.

 The reason that I did all the cut and pasting for the OENavigator component
 instead of just inheriting is that the DBNavigator is part of the BDE Component
 set.  I did not want any of the BDE api getting linked in, since OE replaces the
 BDE.
}

interface
uses SysUtils, Windows, Messages, Classes, Controls, Forms,Dialogs,
  Graphics, Menus, StdCtrls, ExtCtrls, Mask, Buttons, OCIH, OCI, ODI, OCL;

const
  InitRepeatPause = 800;  { pause before repeat timer (ms) }
  RepeatPause     = 600;  { pause before hint window displays (ms)}
  SpaceSize       =  5;   { size of space between special buttons }

type
  EFetchEvent = procedure (Sender: TObject) of object;
  EBeforeUpdateEvent = procedure (Sender: TObject; var bUpdate: Boolean) of object;
  THstmtOperations = (oDelete, oUpdate, oRefresh, oInsert);
  EErrorEvent = procedure (Sender: TObject; Msg: String; Operation: THstmtOperations) of object;
  TOnInsertEvent = procedure (Sender: TObject; var iContinue: Boolean) of object;

  TOEExtHstmt = class(THstmt)
  private
    FAfterPrior:   EFetchEvent;
    FAfterNext:    EFetchEvent;
    FAfterFirst:   EFetchEvent;
    FAfterLast:    EFetchEvent;
    FAfterCancel:  EFetchEvent;
    FAfterPost:    EFetchEvent;
    FAfterRefresh: EFetchEvent;
    FAfterDelete:  EFetchEvent;
    FBeforeCancel: EBeforeUpdateEvent;
    FBeforePost:   EBeforeUpdateEvent;
    FBeforeRefresh:EBeforeUpdateEvent;
    FBeforeDelete: EBeforeUpdateEvent;
    FOnError:      EErrorEvent;
    FOnInsert: TOnInsertEvent;
    FAfterInsert: TNotifyEvent;
  protected
    procedure ePrior;
    procedure eNext;
    procedure eFirst;
    procedure eLast;
    procedure eCancel;
    procedure ePost;
    procedure eRefresh;
    procedure eDelete;
    procedure eInsert(var iContinue: Boolean);
  public
    constructor Create(Owner: TComponent); override;
{    function execute: Boolean; override;}
  published
    property AfterPrior:   EFetchEvent read FAfterPrior write FAfterPrior;
    property AfterNext:    EFetchEvent read FAfterNext write FAfterNext;
    property AfterFirst:   EFetchEvent read FAfterFirst write FAfterFirst;
    property AfterLast:    EFetchEvent read FAfterLast write FAfterLast;
    property AfterCancel:  EFetchEvent read FAfterCancel write FAfterCancel;
    property AfterPost:    EFetchEvent read FAfterPost write FAfterPost;
    property AfterRefresh: EFetchEvent read FAfterRefresh write FAfterRefresh;
    property AfterDelete:  EFetchEvent read FAfterDelete write FAfterDelete;
    property BeforeCancel: EBeforeUpdateEvent read FBeforeCancel write FBeforeCancel;
    property BeforePost:   EBeforeUpdateEvent read FBeforePost write FBeforePost;
    property BeforeRefresh:EBeforeUpdateEvent read FBeforeRefresh write FBeforeRefresh;
    property BeforeDelete: EBeforeUpdateEvent read FBeforeDelete write FBeforeDelete;
    property OnError: EErrorEvent read FOnError Write FOnError;
    property OnInsert: TOnInsertEvent read FOnInsert write FOnInsert;
    property AfterInsert: TNotifyEvent read FAfterInsert write FAfterInsert;
{    property BOF: Boolean  read FBOF;
    property EOF: Booelan  read FEOF;}
  end;

  TOENavButton = class;
  TOENavDataLink = class;

  TNavGlyph = (ngEnabled, ngDisabled);
{  TNavigateBtn = (nbFirst, nbPrior, nbNext, nbLast,
                  nbInsert, nbDelete{, nbEdit}{, nbPost, nbCancel, nbRefresh);}
  TNavigateBtn = (nbFirst, nbPrior, nbNext, nbLast,
                  nbInsert, nbDelete{, nbEdit}, nbPost, nbCancel, nbRefresh);

  TButtonSet = set of TNavigateBtn;
  TNavButtonStyle = set of (nsAllowTimer, nsFocusRect);

  ENavClick = procedure (Sender: TObject; Button: TNavigateBtn) of object;

{ TOEDBNavigator }

  TOEDBNavigator = class (TCustomPanel)
  private
    FDataLink: TOENavDataLink;
    FVisibleButtons: TButtonSet;
    FHints: TStrings;
    ButtonWidth: Integer;
    MinBtnSize: TPoint;
    FOnNavClick: ENavClick;
    FocusedButton: TNavigateBtn;
    FConfirmDelete: Boolean;
//    FOnInsert: TOnInsertEvent;
    FInserting: Boolean;
    FReturnToBrowse: TNotifyEvent;
    function GetDataSource: TOEExtHstmt;
    procedure SetDataSource(Value: TOEExtHstmt);
    procedure InitButtons;
    procedure InitHints;
    procedure Click(Sender: TObject);
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure SetHints(Value: TStrings);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
  protected
    Buttons: array[TNavigateBtn] of TOENavButton;
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure GetChildren(Proc: TGetChildProc); override;
    procedure DoInsert;virtual;
    procedure DoBrowse;virtual;
    procedure DoMove(Index: TNavigateBtn);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TNavigateBtn);
    {MUST Be set at run time}
    property DataSource: TOEExtHstmt read GetDataSource write SetDataSource;
  published
    property VisibleButtons: TButtonSet read FVisibleButtons write SetVisible
      default [nbFirst, nbPrior, nbNext, nbLast, nbInsert, nbDelete{,
        nbEdit}, nbPost, nbCancel, nbRefresh];
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Ctl3D;
    property Hints: TStrings read FHints write SetHints;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick: ENavClick read FOnNavClick write FOnNavClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    property OnStartDrag;
//    property OnInsert: TOnInsertEvent read FOnInsert write FOnInsert;
//    property AfterInsert: TNotifyEvent read FAfterInsert write FAfterInsert;
    property Inserting: Boolean read FInserting write FInserting;
    property OnReturnToBrowse: TNotifyEvent read FReturnToBrowse write FReturnToBrowse;
  end;

{ TNavButton }

  TOENavButton = class(TSpeedButton)
  private
    FIndex: TNavigateBtn;
    FNavStyle: TNavButtonStyle;
    FRepeatTimer: TTimer;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    destructor Destroy; override;
    property NavStyle: TNavButtonStyle read FNavStyle write FNavStyle;
    property Index : TNavigateBtn read FIndex write FIndex;
  end;

{ TOENavDataLink }

  TOENavDataLink = class(TComponent)
  private
    FNavigator: TOEDBNavigator;
    FHstmt: TOEExtHstmt;
    function GetActive: Boolean;
  public
    constructor Create(ANav: TOEDBNavigator);
    destructor Destroy; override;
    property DataSource: TOEExtHstmt read FHstmt write FHstmt;
    property Active: Boolean read GetActive;
  end;

procedure Register;

implementation

{$R OENAV.RES}

{ TOEDBNavigator }

const
  SFirstRecord = 58500;
  SPriorRecord = 58501;
  SNextRecord = 58502;
  SLastRecord = 58503;
  SInsertRecord = 58504;
  SDeleteRecord = 58505;
  {SEditRecord = 58506;}
  SPostEdit = 58506;
  SCancelEdit = 58507;
  SRefreshRecord = 58508;
  SDeleteRecordQuestion = 58609;

  BtnStateName: array[TNavGlyph] of PChar = ('EN', 'DI');
  BtnTypeName: array[TNavigateBtn] of PChar = ('FIRST', 'PRIOR', 'NEXT',
    'LAST', 'INSERT', 'DELETE'{, 'EDIT'}, 'POST', 'CANCEL', 'REFRESH');
  BtnHintId: array[TNavigateBtn] of Word = (SFirstRecord, SPriorRecord,
    SNextRecord, SLastRecord, SInsertRecord, SDeleteRecord{, SEditRecord},
    SPostEdit, SCancelEdit, SRefreshRecord);

Procedure Register;
begin
 RegisterComponents('ODBCExpress', [TOEExtHstmt]);
 RegisterComponents('ODBCExpress', [TOEDBNavigator]);
end;


{ TOEExtHstmt }
constructor TOEExtHstmt.Create(Owner: TComponent);
begin
 inherited Create(Owner);
 ExtendedFetch := True;
 CursorType := SQL_CURSOR_DYNAMIC;
{ FBOF := False;
 FEOF := False;}
end;

{function TExtHstmt.execute: Boolean;
begin
 Result := inherited execute;
 if Result then begin
  FEOF := False;
  FBOF := True;
 end
 else begin
  FEOF := False;
  FBOF := False;
 end;
end;
 }
procedure TOEExtHstmt.ePrior;
begin
{ FBOF := False; FEOF := False;}
 if FetchPrev then begin
    if Assigned(FAfterPrior) then FAfterPrior(Self);
  end
 else {must have been at first, therefore reset to first}
    FetchFirst;
end;

procedure TOEExtHstmt.eNext;
begin
 if FetchNext then begin
   if Assigned(FAfterNext) then FAfterNext(Self);
 end
else {must have been at end, reset to last}
  FetchLast;
end;

procedure TOEExtHstmt.eFirst;
begin
 if FetchFirst then
  if Assigned(FAfterFirst) then FAfterFirst(Self);
end;

procedure TOEExtHstmt.eLast;
begin
 if FetchLast then
  if Assigned(FAfterLast) then FAfterLast(Self);
end;

procedure TOEExtHstmt.eCancel;
var
 tContinue: Boolean;
begin
 tContinue := True;
 if Assigned(FBeforeCancel) then FBeforeCancel(Self, tContinue);
 if Not tContinue then Exit;
TRY
 if DoRefresh then
  if Assigned(FAfterCancel) then FAfterCancel(Self);
EXCEPT
 on ODBCError do begin
    if Assigned(FOnError) then begin
     Hdbc.Henv.Error.GetNext;
     FOnError(Self, Hdbc.Henv.Error.State + ': ' + Hdbc.Henv.Error.Msg, oRefresh);
    end;
 end;
END;
end;

procedure TOEExtHstmt.ePost;
var
 tContinue: Boolean;
begin
 tContinue := True;
 if Assigned(FBeforePost) then FBeforePost(Self, tContinue);
 if Not tContinue then Exit;
TRY
 if DoUpdate then begin
    if Assigned(FAfterPost) then FAfterPost(Self);
 end;
EXCEPT
 on ODBCError do begin
    if Assigned(FOnError) then begin
     Hdbc.Henv.Error.GetNext;
     FOnError(Self, Hdbc.Henv.Error.State + ': ' + Hdbc.Henv.Error.Msg, oUpdate);
    end;
 end;
END;
end;

procedure TOEExtHstmt.eRefresh;
var
 tContinue: Boolean;
begin
 tContinue := True;
 if Assigned(FBeforeRefresh) then FBeforeRefresh(Self, tContinue);
 if Not tContinue then Exit;
TRY
 ePost;{post any changed}
 if DoRefresh then
  if Assigned(FAfterCancel) then FAfterCancel(Self);
EXCEPT
 on ODBCError do begin
    if Assigned(FOnError) then begin
     Hdbc.Henv.Error.GetNext;
     FOnError(Self, Hdbc.Henv.Error.State + ': ' + Hdbc.Henv.Error.Msg, oRefresh);
    end;
 end;
END;
end;

procedure TOEExtHstmt.eDelete;
var
 tContinue: Boolean;
begin
 tContinue := True;
 if Assigned(FBeforeDelete) then FBeforeDelete(Self, tContinue);
 if Not tContinue then Exit;
 TRY
 if DoDelete then begin
  if Not FetchNext then begin
     FetchPrev;{ put it back }
   {  FetchPrev;{ try to put it to a valid record before the one that was deleted}
  end;
  if Assigned(FAfterDelete) then FAfterDelete(Self);
 end;
 EXCEPT
 on ODBCError do begin
    if Assigned(FOnError) then begin
     Hdbc.Henv.Error.GetNext;
     FOnError(Self, Hdbc.Henv.Error.State + ': ' + Hdbc.Henv.Error.Msg, oDelete);
    end;
 end;
END;
end;

procedure TOEExtHstmt.eInsert(var iContinue: Boolean);
begin
 iContinue := False;
TRY
  if Assigned(FOnInsert) then
   FOnInsert(Self, iContinue)
  else
   Exit; {If the user hasn't assigned anything to the insert then exit}
  if Not iContinue then Exit;
  if iContinue then begin
    If Assigned(FAfterInsert) then FAfterInsert(Self);
  end;
EXCEPT
 on ODBCError do begin
    if Assigned(FOnError) then begin
     Hdbc.Henv.Error.GetNext;
     FOnError(Self, Hdbc.Henv.Error.State + ': ' + Hdbc.Henv.Error.Msg, oInsert);
    end;
 end;
END;
end;

constructor TOEDBNavigator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  {!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!}
  FDataLink := TOENavDataLink.Create(Self);
  FVisibleButtons := [nbFirst, nbPrior, nbNext, nbLast, nbInsert,
    nbDelete{, nbEdit}, nbPost, nbCancel, nbRefresh];
  FHints := TStringList.Create;
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 241;
  Height := 25;
  ButtonWidth := 0;
  FocusedButton := nbFirst;
  FConfirmDelete := True;
  Inserting := False;
end;

destructor TOEDBNavigator.Destroy;
begin
  FDataLink.Free;
  FHints.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TOEDBNavigator.InitButtons;
var
  I: TNavigateBtn;
  Btn: TOENavButton;
  X: Integer;
  ResName: string;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TOENavButton.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    FmtStr(ResName, 'oen_%s', [BtnTypeName[I]]);
    Btn.Glyph.Handle := LoadBitmap(HInstance, PChar(ResName));
    Btn.NumGlyphs := 2;
    Btn.Enabled := False;  {!!! Force creation of speedbutton images !!!}
    Btn.Enabled := True;
    Btn.OnClick := Click;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
  InitHints;
  Buttons[nbPrior].NavStyle := Buttons[nbPrior].NavStyle + [nsAllowTimer];
  Buttons[nbNext].NavStyle  := Buttons[nbNext].NavStyle + [nsAllowTimer];
end;

procedure TOEDBNavigator.InitHints;
var
  I: Integer;
  J: TNavigateBtn;
begin
  for J := Low(Buttons) to High(Buttons) do
    Buttons[J].Hint := LoadStr (BtnHintId[J]);
  J := Low(Buttons);
  for I := 0 to (FHints.Count - 1) do
  begin
    if FHints.Strings[I] <> '' then Buttons[J].Hint := FHints.Strings[I];
    if J = High(Buttons) then Exit;
    Inc(J);
  end;
end;

procedure TOEDBNavigator.SetHints(Value: TStrings);
begin
  FHints.Assign(Value);
  InitHints;
end;

procedure TOEDBNavigator.GetChildren(Proc: TGetChildProc);
begin
end;

procedure TOEDBNavigator.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TOEDBNavigator.SetVisible(Value: TButtonSet);
var
  I: TNavigateBtn;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TOEDBNavigator.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TNavigateBtn;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * MinBtnSize.X;
  if W < MinW then W := MinW;
  if H < MinBtnSize.Y then H := MinBtnSize.Y;

  ButtonWidth := W div Count;
  Temp := Count * ButtonWidth;
  if Align = alNone then W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec(Temp, Remain);
        if Temp < 0 then
        begin
          Inc(Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds(X, 0, ButtonWidth + Space, Height);
      Inc(X, ButtonWidth + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TOEDBNavigator.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TOEDBNavigator.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TOEDBNavigator.Click(Sender: TObject);
begin
  BtnClick (TOENavButton (Sender).Index);
end;

procedure TOEDBNavigator.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TNavigateBtn;
begin
  OldFocus := FocusedButton;
  FocusedButton := TOENavButton (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TOEDBNavigator.BtnClick(Index: TNavigateBtn);
{OVERRIDE THIS>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>}
begin
  {if (DataSource <> nil) and (DataSource.State <> dsInactive) then}
  if (DataSource <> nil) and (DataSource.Active) then
  begin
   {with DataSource.DataSet do}
   with DataSource do
    begin
      case Index of
        nbPrior: DoMove(Index);
        nbNext: DoMove(Index);
        nbFirst: DoMove(Index);
        nbLast: DoMove(Index);
        nbInsert: DoInsert;{this is a method of TOEDBNavigator and NOT OEExtHstmt}
        {nbEdit: Edit;}
        nbCancel: eCancel;
        nbPost: ePost;
        nbRefresh: eRefresh;
        nbDelete:
          if not FConfirmDelete or
            (MessageDlg(LoadStr(SDeleteRecordQuestion), mtConfirmation,
            mbOKCancel, 0) <> idCancel) then eDelete;
      end;{case}
    end;
  end;
  if not (csDesigning in ComponentState) and Assigned(FOnNavClick) then
    FOnNavClick(Self, Index);
end;

procedure TOEDBNavigator.DoMove(Index: TNavigateBtn);
begin
 if Inserting then DoBrowse;
 case Index of
  nbPrior: DataSource.ePrior;
  nbNext: DataSource.eNext;
  nbFirst: DataSource.eFirst;
  nbLast: DataSource.eLast;
 end;{case}
end;

procedure TOEDBNavigator.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TOEDBNavigator.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TOEDBNavigator.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TNavigateBtn;
  OldFocus: TNavigateBtn;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TOEDBNavigator.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TOEDBNavigator.DoInsert;
var
 iContinue: Boolean;
begin
 iContinue := True;
 DataSource.eInsert(iContinue);
 Inserting := iContinue;
end;

procedure TOEDBNavigator.DoBrowse;
begin
  Inserting := False;
  If Assigned(FReturnToBrowse) then FReturnToBrowse(Self);
end;

procedure TOEDBNavigator.DataChanged;
var
  UpEnable, DnEnable: Boolean;
begin
{!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!} {Changed FDataLink.DataSet to
  FDataLink.DataSource}
  UpEnable := Enabled and FDataLink.Active{ and not FDataLink.DataSource.BOF};
  DnEnable := Enabled and FDataLink.Active{ and not FDataLink.DataSource.EOF};
  Buttons[nbFirst].Enabled := UpEnable;
  Buttons[nbPrior].Enabled := UpEnable;
  Buttons[nbNext].Enabled := DnEnable;
  Buttons[nbLast].Enabled := DnEnable;
  Buttons[nbDelete].Enabled := Enabled and FDataLink.Active and
    (DataSource.ConcurrencyType <> SQL_CONCUR_READ_ONLY)
   { FDataLink.DataSet.CanModify and
    not (FDataLink.DataSource.BOF and FDataLink.DataSource.EOF)};
end;

procedure TOEDBNavigator.EditingChanged;
{OVERRIDE, Makes reference to TQuery!}
var
  CanModify: Boolean;
begin
  CanModify := Enabled and FDataLink.Active and (DataSource.ConcurrencyType <> SQL_CONCUR_READ_ONLY)
  { and FDataLink.DataSet.CanModify};
  Buttons[nbInsert].Enabled := Assigned(DataSource.OnInsert){CanModify};
  {Buttons[nbEdit].Enabled := CanModify and not FDataLink.Editing;}
  Buttons[nbPost].Enabled := CanModify{ and FDataLink.Editing};
  Buttons[nbRefresh].Enabled := CanModify{ and FDataLink.Editing};
  Buttons[nbCancel].Enabled := CanModify;
//  Buttons[nbRefresh].Enabled := CanModify;{not (FDataLink.DataSet is TQuery);}
end;

procedure TOEDBNavigator.ActiveChanged;
var
  I: TNavigateBtn;
begin
  if not (Enabled and FDataLink.Active) then
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Enabled := False
  else
  begin
    DataChanged;
    EditingChanged;
  end;
end;

procedure TOEDBNavigator.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TOEDBNavigator.SetDataSource(Value: TOEExtHstmt);
begin
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TOEDBNavigator.GetDataSource: TOEExtHstmt;
begin
  Result := FDataLink.DataSource;
end;

procedure TOEDBNavigator.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitHints;
  ActiveChanged;
end;

{TOENavButton}

destructor TOENavButton.Destroy;
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy;
end;

procedure TOENavButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  if nsAllowTimer in FNavStyle then
  begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);

    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True;
  end;
end;

procedure TOENavButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
                                  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled  := False;
end;

procedure TOENavButton.TimerExpired(Sender: TObject);
begin
  FRepeatTimer.Interval := RepeatPause;
  if (FState = bsDown) and MouseCapture then
  begin
    try
      Click;
    except
      FRepeatTimer.Enabled := False;
      raise;
    end;
  end;
end;

procedure TOENavButton.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TOEDBNavigator (Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

{ TOENavDataLink }

constructor TOENavDataLink.Create(ANav: TOEDBNavigator);
begin
  inherited Create(ANav);
  FNavigator := ANav;
end;

destructor TOENavDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy;
end;

function TOENavDataLink.GetActive;
begin
 if Assigned(DataSource) then
  Result := DataSource.Active
 else
  Result := False;
end;
{
procedure TNavDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged;
end;

procedure TNavDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged;
end;

procedure TNavDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged;
end;
}
end.
