unit oddscb;
{
 Three slightly useful components.  I had a form where I wanted to let the user
 specify which datasources(DataSource, Table, Column) to use to retrieve data
 from, so I looked at the ODBC examples and found it was rather easy to list
 the datasources.  Then thought, hey this is Delphi, lets make a component.  It
 started out as a simple Datasource lister, then I made the second component,
 the TreeView of the Tables-Columns, Indexes... of the selected DataSource of
 the DataSource List.

 Either componenet can be used alone or together.  When used together automatic
 updating of the table/column component can be done, so that when ever the
 DataSource lister index changes, it will automatically update the table/column
 view.

 TODBCDataSources Component
 Properties
  Henv: THenv - this is the ODBC Environment handle to use to make the necessary
                ODBC API call to retrieve the datasources.  When this property is
                set the list is automatically populated.

  DataSource  - Set or Return the datasource in the list.  If you try to set the
                list index to a DataSource that does not exist, an exception is raised
                EODBCNoSuchDataSource.  This exception descends from the ODBC
                Express exception class.

  TableList   - Drop a TODBCTables component on the same form, set AutoUpdate to
                true, and bingo the TODBCTables gets filled with the datasources
                tables/columns.

  AutoUpdate  - Boolean, Set to true to update the attached (if any) TODBCTables



 Methods
  Create      - Just sets some of the combobox properties, like sorted and list type.

  Populate    - You DO NOT need to call this method, it is only exposed so that you
                may call it if you need to.  When would you need to?  If you think
                that the user may have added an ODBC datasource since you created
                the form.

------

 TODBCTables
 Properties
  Henv        - ODBC Environment handle
  Table       - Currently Selected table in the list(even if the selected item is
                a column)
  Column      - Returns the selected Column, if any.

  Text        - If you don't care if a column or table then.... Improvement over
                TTreeView.Selected.Text, checks for nil pointer first.

  DataSource  - A String that specifies the ODBC Datasource to connect to, and retreive
                the table names/columns from. If using AutoUpdate of the TODBCDataSources
                component, this gets done automatically.

  FullConnectString - The component does not use the Hdbc component of the ODBCExpress
                library, instead it uses the SQLDriverConnect function, to allow
                for interactive logins (get name/password...).  This way the component
                does not have to worry about prompting for such data, and can
                leave it to the driver.

  ShowColumns: - While I only needed to display the columns, I thought I would make
                 the component generic enough so that if one wished, they could display
                 all sorts of info about tables.  Set to true to display column names.

  ShowIndexes: - This was included to show how to do something other than Columns. Set
                 It to true to see the indexes contained in each of the Tables.

 Methods
  Populate    - Again, there should be no need to call this method, it is only exposed
                so that you can force updates.

  Create      - Needed to create the TImageList component to get the bmp for the
                view.  As well as set the component to readonly.

  Destroy     - Free the imagelist....

 TOEHexEdit - Credit goes to Immo Wache, Rostock Deutschland,
              immo.wache@mbst.uni-rostock.de
              for this component. Immo had created a TEdit component which restricted
              entires to only Integers, in Hex, Octal or Decimal format. I changed the
              base class to TOEEdit and added an addtional property to allow entry of
              integer strings larger then the LongInt max. Thanks Immo!
------
 Greg Carter
 CRYPTOCard Corporation
 July 18 1996

 Suggested Improvemetns: Allow to dynamically update the list, so that when a user
 clicks on the 'Columns' tree the columns are then enumerated.  Right now all
 enumeration occures at the time of connect. So it can be slow for large dbs.
 This make be as easy as ovrriding the click handler.

--------------------------------------------------------------------------------
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, StdCtrls, OCIH, OCI, ODI, OCL,
  ComCtrls, Forms, Menus, RichEdit, CommCtrl;
Type

 TODBCTables = class;{forward declare so we can use it in TODBCDataSources}

 TODBCDataSources = class(TComboBox)
  private
   FHenv : THenv;
   FTableList : TODBCTables;
   FAutoUpdate: Boolean;
   procedure SetHenv(Henv: THenv);
   function  GetDataSource: String;
   procedure SetDataSource(DataSrc: String);
   procedure Change;override;
  public
   procedure Populate;
   constructor Create(Owner: TComponent); override;
  published
   property Henv: THenv Read FHenv Write SetHenv;
   property DataSource: String Read GetDataSource Write SetDataSource;
   property TableList: TODBCTables Read FTableList Write FTableList;
   property AutoUpdate: Boolean Read FAutoUpdate Write FAutoUpDate;
 end;

 EODBCNoHenv = class(ODBCError);
 EODBCNoSuchDataSource = class(ODBCError);
 EODBCNoTables = class(ODBCError);

 TODBCTables = class(TTreeView)
  private
   FImageList : TImageList;
   FHenv : THenv;
   FItemHstmt: THstmt;
   FDataSource : String;
   FFullString : String;
   FShowColumns, FShowIndexes: Boolean;
   procedure SetDataSource(DataSrc: String);
   function  GetTable: String;
   function  GetColumn: String;
   function  GetText: String;
   procedure PutColumns(Const ParentNode: TTreeNode; Const TableName: String);
   procedure PutIndexes(Const ParentNode: TTreeNode; Const TableName: String);
  public
   procedure Populate;
   constructor Create(Owner: TComponent); override;
   destructor Destroy; override;
   function isODBCColumn(const Node: TTreeNode): Boolean;
  published
   property Henv: THenv Read FHenv Write FHenv;
   property Table: String Read GetTable stored False;
   property Column: String Read GetColumn stored False;
   property Text: String Read GetText stored False;
   property DataSource: String Read FDataSource Write SetDataSource stored False;
   property FullConnectString: String Read FFullString stored False;
   property ShowColumns: Boolean Read FShowColumns Write FShowColumns;
   property ShowIndexes: Boolean Read FShowIndexes Write FShowIndexes;
 end;

 EODBCNoSuchTable = class(ODBCError);
 TEditBase =(ebHex, ebDec, ebOct, ebBin);

 TOEHexEdit = class(TOEEdit)
   private
    FMinValue, FMaxValue: Longint;
    FValidate, FValueCheck: Boolean;
    FNumBase: TEditBase;
    procedure SetNumBase( NewValue: TEditBase);
    procedure SetValue( NewValue : Longint );
    function  GetValue: Longint;
    function  CheckValue( NewValue: Longint): Longint;
    procedure SetMaxValue( NewValue: Longint);
    procedure SetMinValue( NewValue: Longint);
    procedure SetValidate( B:  Boolean);
    function  SyntaxOk(const S: string): Boolean;
    procedure CMEnter(var Message: TCMGotFocus); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    function  BaseStrToInt(const S: string): Longint;
  protected
    function  IsValidChar( Key: Char) : Boolean; virtual;
    procedure KeyDown(var Key: Word; Shift : TShiftState ); override;
    procedure KeyPress(var Key: Char ); override;
    function  ValidCopy: Boolean;
    function  ValidPaste: Boolean;
    function  ValidCut: Boolean;
    function  ValidDelete: Boolean;
  public
    function  IntToBaseStr(Value: Longint): string;
    constructor Create( AOwner : TComponent ); override;
  published
    property NumBase: TEditBase read FNumBase write SetNumBase;
    property MaxValue: Longint read FMaxValue write SetMaxValue;
    property MinValue: Longint read FMinValue write SetMinValue;
    property Validate: Boolean read FValidate write SetValidate;
{ Set CheckValue to true if you only want hex numbers that are within the LongInt
  range.  Set to false if all you want is hex strings(which may represent numbers
  larger then Max(LongInt) }
    property ValueCheck: Boolean read FValueCheck write FValueCheck;
    property Value: Longint read GetValue write SetValue;
  end;

 procedure Register;

implementation
uses Clipbrd;
Const
itBitmap: TResType = Controls.rtBitmap; // Reference for duplicate identifier
COLUMNTEXT = 'Columns';
INDEXTEXT = 'Indexes';
Base: array[TEditBase] of Byte =(16, 10, 8, 2);

{$R ODTABLE.RES} {Res file with bitmaps for TODBCTables Tree View}

procedure Register;
begin
 RegisterComponents('ODBCExpress', [TODBCDataSources]);
 RegisterComponents('ODBCExpress', [TODBCTables]);
 RegisterComponents('ODBCExpress', [TOEHexEdit]);
end;
{ TOEHexEdit }

constructor TOEHexEdit.Create( AOwner : TComponent);
begin
  inherited Create( AOwner);
  {force Text to '0'}
  SetValue( 0);
end;

function TOEHexEdit.BaseStrToInt(const S: string): Longint;
var
  Digit, I: Byte;
begin
  Result :=0;
  for I :=1 to Length( S) do
  begin
    Digit :=ord( S[I]) -ord('0');
    if Digit >10 then Dec( Digit, 7);
    Result :=Result *Base[NumBase] +Digit;
  end;
end;

function TOEHexEdit.IntToBaseStr(Value: Longint): string;
var
  Ch: Char;
begin
  Result :='';
  repeat
    Ch :='0';
    Inc(Ch, Value mod Base[NumBase]);
    if Ch >'9' then Inc(Ch, 7);
    Insert( Ch, Result, 1);
    Value :=Value div Base[NumBase];
  until Value =0;
end;

function TOEHexEdit.GetValue: Longint;
begin
  Result :=BaseStrToInt( Text);
end;

procedure TOEHexEdit.SetValue(NewValue: Longint);
begin
if ValueCheck then
   Text:=IntToBaseStr( CheckValue( NewValue));
end;

procedure TOEHexEdit.SetNumBase( NewValue: TEditBase);
var
  TempValue: LongInt;
begin
  TempValue :=Value;
  FNumBase :=NewValue;
  SetValue( TempValue);
end;

function TOEHexEdit.CheckValue( NewValue: Longint): Longint;
begin
  if NewValue <0 then NewValue :=0;
  Result :=NewValue;
  if FValidate then
  begin
    if      NewValue < FMinValue then Result:= FMinValue
    else if NewValue > FMaxValue then Result:= FMaxValue;
  end;
end;

procedure TOEHexEdit.SetMaxValue( NewValue: Longint);
begin
  if NewValue <0 then NewValue :=0;
  FMaxValue := NewValue;
  SetValue( Value);
end;

procedure TOEHexEdit.SetMinValue( NewValue: Longint);
begin
  if NewValue <0 then NewValue :=0;
  FMinValue := NewValue;
  SetValue( Value);
end;

procedure TOEHexEdit.SetValidate( B:  Boolean);
begin
  FValidate :=B;
  SetValue( Value);
end;

function TOEHexEdit.SyntaxOk(const S: string): Boolean;
var
  Digit:    Byte;
  I:        Byte;
  NewValue: LongInt;
begin
  { syntax correct if all chars are valid }
  Result := True;
  for I :=1 to Length(S) do
  begin
     if not (S[I] in ['0'..'9', 'A'..'F']) or
        (BaseStrToInt( S[I]) >=Base[NumBase]) then Result := False;
  end;
  { syntax correct if Value inside bounds }
  if Result and FValidate then
  begin
    NewValue :=BaseStrToInt( S);
    if (NewValue < FMinValue) or (NewValue > FMaxValue) then Result:= False;
  end;
end;

procedure TOEHexEdit.CMEnter( var Message: TCMGotFocus);
begin
  if AutoSelect and not( csLButtonDown in ControlState ) then SelectAll;
  inherited;
end;

procedure TOEHexEdit.CMExit( var Message: TCMExit);
begin
  inherited;
  SetValue( Value);
end;

function TOEHexEdit.IsValidChar( Key: Char): Boolean;
begin
  case Key of
    '0'..'9', 'A'..'F':
         Result := SyntaxOk( Copy( Text, 1, SelStart ) +
                             Key +
                             Copy( Text, SelStart+1+SelLength, 255 ));
    ^H: if SelLength = 0 then                         { ^H = Backspace }
          Result := SyntaxOk( Copy( Text, 1, SelStart-1 ) +
                              Copy( Text, SelStart+1, 255 ))
        else
          Result := SyntaxOk( Copy( Text, 1, SelStart ) +
                              Copy( Text, SelStart+1+SelLength, 255 ));
  else
    Result := False;
  end;{case}
end;

function TOEHexEdit.ValidCopy: Boolean;
begin
  Result :=True;
end;

function TOEHexEdit.ValidPaste: Boolean;
begin
  if Clipboard.HasFormat(CF_TEXT) then
    Result :=SyntaxOk( Copy( Text, 1, SelStart ) +
                       Clipboard.AsText +
                       Copy( Text, SelStart+1, 255 ));
end;

function TOEHexEdit.ValidCut: Boolean;
begin
  Result :=SyntaxOk( Copy( Text, 1, SelStart ) +
                     Copy( Text, SelStart+1+SelLength, 255 ));
end;

function TOEHexEdit.ValidDelete: Boolean;
var
  S: string;
begin
  if SelLength =0 then
    S :=Copy( Text, 1, SelStart ) +Copy( Text, SelStart+2, 255 )
  else
    S :=Copy( Text, 1, SelStart ) +Copy( Text, SelStart+1+SelLength, 255 );
  Result :=SyntaxOk( S);
end;

procedure TOEHexEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  { handle Copy-, Paste-, Cut-, Delete-Keys}
  if ssShift in Shift then
  begin
    if      Key =VK_INSERT then begin if not ValidPaste then Key :=0 end
    else if Key =VK_DELETE then begin if not ValidCut   then Key :=0 end
  end
  else if ssCtrl in Shift then
  begin
    if      Key =VK_INSERT then begin if not ValidCopy then Key :=0 end
    else if Key =VK_DELETE then begin if not ValidDelete then Key :=0 end
  end
  else if Key =VK_DELETE then begin if not ValidDelete then Key :=0 end;
  inherited KeyDown( Key, Shift);
end;

procedure TOEHexEdit.KeyPress(var Key: Char);
begin
  { handle Copy Paste Cut Keys}
  if      Key =^C then begin if not ValidCopy  then Key :=#0 end
  else if Key =^V then begin if not ValidPaste then Key :=#0 end
  else if Key =^X then begin if not ValidCut   then Key :=#0 end
  else
  begin
    Key :=UpCase( Key);  {transform a..f to A..F}
    if not IsValidChar( Key) then Key := #0;
  end;
  if Key <>#0 then inherited KeyPress( Key);
end;


{ TODBCTables }

constructor TODBCTables.Create(Owner: TComponent);
begin
 inherited Create(Owner);
 ReadOnly := True;
 FShowColumns := True;
 FShowIndexes := True;
 if Not (csDesigning in ComponentState) then begin
    FImageList := TImageList.CreateSize(16, 17);
 {Note if you change the bitmaps used you will probably need to change the
 CreateSize parameters}
    FImageList.ResourceLoad(itBitmap, 'ListImages', clOlive);
    Images := FImageList;
    FItemHstmt := THstmt.Create(Owner);
 end;   
end;

destructor TODBCTables.Destroy;
begin
 FImageList.Free;
 FItemHstmt.Free;
inherited Destroy;
end;

procedure TODBCTables.SetDataSource(DataSrc: String);
begin
 FDataSource := DataSrc;
 if Assigned(FHenv) And (Not (csDesigning in ComponentState)) then
   Populate;
end;

function  TODBCTables.GetTable: String;
var
 Node: TTreeNode;
begin
 Node := Selected;
 Result := '';
 while (Node <> nil) And (Node.Level > 0) do {loop until get to top}{check for nil?}
       Node := Node.Parent;
 if Node <> nil then Result := Node.Text;
end;

function TODBCTables.GetColumn: String;
var
 Node: TTreeNode;
begin
 Node:= Selected;
 Result := '';
 if  Node = nil then exit;
 if isODBCColumn(Node) then
    Result := Node.Text;
end;

function  TODBCTables.GetText: String;
begin
 if Selected <> nil then
  Result := Selected.Text
 else
  Result := '';
end;

function TODBCTables.isODBCColumn(const Node: TTreeNode): Boolean;
var
 tNode: TTreeNode;
begin
 tNode := Node.Parent;
 if (tNode <> nil) and (tNode.Text = COLUMNTEXT) then
    Result := True
 else
    Result := False;
end;

procedure TODBCTables.PutColumns(const ParentNode: TTreeNode; Const TableName: String);
var
 ColNode, ChildNode: TTreeNode;
 ret : SQLRETURN;
begin
 {add the 'Columns' node, then add its child nodes}
 ColNode := Items.AddChild(ParentNode, COLUMNTEXT);
 ColNode.ImageIndex := 1;              {setup various image indexes}
 ColNode.StateIndex := 1;
 ColNode.SelectedIndex := 1;

 FItemHstmt.Terminate;
  // ret := _SQLColumns(FhstmtCol.Handle, CatName, SchName, TableName, ColName);
 ret := SQLColumns(FItemHstmt.Handle, nil, 0, nil, 0, Pointer(PChar(TableName)), SQL_NTS,nil, 0);
 if Not FHenv.Error.Success(ret) then  {couldn't get column names, go around}
    exit;
 while FItemHstmt.FetchNext do begin   {add column names to view}
    ChildNode := Items.AddChild(ColNode, FItemHstmt.ColString(4));
    ChildNode.ImageIndex := 1;         {setup various image indexes}
    ChildNode.StateIndex := 1;
    ChildNode.SelectedIndex := 1;
 end;
end;

procedure TODBCTables.PutIndexes(const ParentNode: TTreeNode; Const TableName: String);
var
 ColNode, ChildNode: TTreeNode;
 ret : SQLRETURN;
begin
 {add the 'Index' node, then add its child nodes}
 ColNode := Items.AddChild(ParentNode, INDEXTEXT);
 ColNode.ImageIndex := 1;              {setup various image indexes}
 ColNode.StateIndex := 1;
 ColNode.SelectedIndex := 1;

 FItemHstmt.Terminate;
 // ret := _SQLStatistics(FhstmtCol.Handle, '', '', TableName, SQL_INDEX_ALL, SQL_QUICK);
 ret := SQLStatistics(FItemHstmt.Handle, nil, 0, nil, 0, Pointer(PChar(TableName)), Length(TableName),SQL_INDEX_ALL, SQL_QUICK);
 if Not FHenv.Error.Success(ret) then  {couldn't get index names, go around}
    exit;
 while FItemHstmt.FetchNext do begin   {add index names to view}
   if (SmallintPtr(FItemHstmt.ColValue(7))^ <> SQL_TABLE_STAT) and
       (SmallintPtr(FItemHstmt.ColValue(8))^ = 1) then begin {only display the index name once}
     ChildNode := Items.AddChild(ColNode, FItemHstmt.ColString(6));
     ChildNode.ImageIndex := 1;        {setup various image indexes}
     ChildNode.StateIndex := 1;
     ChildNode.SelectedIndex := 1;
   end;
 end;
end;

procedure TODBCTables.Populate;
var
 CatName, SchName, TableName, TableType, ColName: String;
 ret : SQLRETURN;
 TablesHstmt: THStmt;
 connectStr: String;
 Tableshdbc: Thdbc;
 CNode, Node : TTreeNode;
 i: SQLINTEGER;
begin
CatName := ''; SchName := ''; TableName := ''; ColName := '';
Items.BeginUpdate;
TablesHstmt := THstmt.Create(Self);
Tableshdbc := Thdbc.Create(Self);
TRY
TRY
   Items.Clear;
   Tableshdbc.Henv := FHenv;
   connectStr := 'DSN=' + FDataSource;
{Future version will use SQLDriver connect}
{   SetLength(FFullString, 500);
   if Not FHenv.Error.Success(_SQLDriverConnect(hdbc1.Handle, Self.Handle, connectStr, FFullString,SQL_DRIVER_COMPLETE)) then
      Raise ODBCError.Create('Unable to Connect to DataSource');
   if ret = SQL_SUCCESS_WITH_INFO then begin
      ret := _SQLError(Fhenv.Handle, hdbc1.Handle, nil, catName, i,SchName);
      FHenv.Error.GetNext;
      FHenv.Error.Display;
      exit;
   end;
 }
   Tableshdbc.DataSource := FDataSource;
   Tableshdbc.Connect;
   TableType := 'TABLE, VIEW, GLOBAL TEMPORARY, LOCAL TEMPORARY, ALIAS, SYNONYM';
   TablesHstmt.Hdbc := TablesHdbc;
   FItemHstmt.Hdbc := TablesHdbc;
   FItemHstmt.Terminate;
 //  ret := _SQLTables(TablesHstmt.Handle, CatName , SchName, TableName, TableType);
   if Not FHenv.Error.Success(SQLTables(TablesHstmt.Handle,nil, 0, nil, 0, nil, 0, Pointer(PChar(TableType)), Length(TableType))) then
      Raise EODBCNoTables.Create('Unable to determine table names');
   Node := nil;
   while TablesHstmt.FetchNext do begin
   {populate the treeview}
      Node := Items.Add(Node, TablesHstmt.ColString(3));{3rd column has table name in it}
      TableName := TablesHstmt.ColString(3);
      If ShowColumns then PutColumns(Node, TableName);
      If ShowIndexes then PutIndexes(Node, TableName);
   end;
   Items.EndUpDate;
EXCEPT
 on ODBCError do
 begin
    FHenv.Error.GetNext;
    FHenv.Error.Display;
 end;
END;
FINALLY
 Items.EndUpDate;
 TablesHstmt.Free;
 Tableshdbc.free;
END;
end;

{ TODBCDataSources }

constructor TODBCDataSources.Create(Owner: TComponent);
begin
 inherited Create(Owner);
 Style :=  csDropDownList;
 Sorted := True;
 FAutoUpdate := True;
 FTableList := nil;
end;

procedure TODBCDataSources.Change;
begin
 Screen.Cursor := crHourglass;
 TRY
 inherited Change;
 if FAutoUpdate And (FTableList <> nil) then
  {set the Datasource of the ODBCTables List which will cause it to update}
    FTableList.DataSource := DataSource;
 FINALLY
  Screen.Cursor := crDefault;
 END;
end;

procedure TODBCDataSources.Populate;
var
 DSName, DSDescr: String;
begin
  if FHenv = nil then
   raise EODBCNoHenv.Create('No ODBC Environment handle specified');

  Clear;
    { Retrieve Data Sources }
  while FHenv.Error.Success(_SQLDataSources(FHenv.Handle, SQL_FETCH_NEXT, DSName, DSDescr)) do
    Items.Add(DSName);
  if Items.Count > 0 then
     ItemIndex := 0;
end;

procedure TODBCDataSources.SetHenv(Henv: THenv);
begin
 FHenv := Henv;
 Populate;
end;

function TODBCDataSources.GetDataSource: String;
begin
 Result := Text;
end;

procedure TODBCDataSources.SetDataSource(DataSrc: String);
var
 index: integer;
begin
  index := Items.IndexOf(DataSrc);
  if index > -1 then
   ItemIndex := index
  else
   raise EODBCNoSuchDataSource.Create('The DataSource "' + DataSrc + '" does not exist');
end;

end.
