unit secureobj;
// TSecureableObject - Implements NT security on Objects.
// Author - Greg Carter - CRYPTOCard Corporation 1996
// Adapted from sec.cpp by Ruediger R. Asche, Microsoft Developer Network
// Technology Group,  from the articles
// Windows NT Security in Theory & Practice
// The Guts of Security  and
// Security Bits and Pieces

// Under NT there are a number of Objects which have inherent security, that is
// the system handles the checks when a user tries to access the object.  Objects
// include Kernel Objects (Mutex, Process, Thread Handles), Files (NTFS only, Not
// FAT), User Objects (windows & menus..), Registry Keys, and Service Objects (NT
// Services).  You may also add security to Private Objects (Things which you make up),
// but you must handle the calls to AccessCheck.

// This base class TSecureable Object implements all the tedious tasks of
// 'playing' with Security Descriptors, ACL, ACE...

interface
uses Windows, Classes, SysUtils, WinSvc;


/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// Universal well-known SIDs                                               //
//                                                                         //
//     Null SID              S-1-0-0                                       //
//     World                 S-1-1-0                                       //
//     Local                 S-1-2-0                                       //
//     Creator Owner ID      S-1-3-0                                       //
//     Creator Group ID      S-1-3-1                                       //
//                                                                         //
//     (Non-unique IDs)      S-1-4                                         //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

Const
SECURITY_NULL_SID_AUTHORITY: array[0..5] of Byte = (0,0,0,0,0,0);
SECURITY_WORLD_SID_AUTHORITY: array[0..5] of Byte = (0,0,0,0,0,1);
SECURITY_LOCAL_SID_AUTHORITY: array[0..5] of Byte = (0,0,0,0,0,2);
SECURITY_CREATOR_SID_AUTHORITY: array[0..5] of Byte = (0,0,0,0,0,3);
SECURITY_NON_UNIQUE_AUTHORITY: array[0..5] of Byte = (0,0,0,0,0,4);

SECURITY_NULL_RID: LongInt = $00000000;// (0x00000000L);
SECURITY_WORLD_RID:LongInt = $00000000;// (0x00000000L)
SECURITY_LOCAL_RID:LongInt = $00000000;// (0X00000000L)

SECURITY_CREATOR_OWNER_RID: LongInt = $00000000;//     (0x00000000L)
SECURITY_CREATOR_GROUP_RID: LongInt = $00000001;//     (0x00000001L)

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// NT well-known SIDs                                                      //
//                                                                         //
//     NT Authority          S-1-5                                         //
//     Dialup                S-1-5-1                                       //
//                                                                         //
//     Network               S-1-5-2                                       //
//     Batch                 S-1-5-3                                       //
//     Interactive           S-1-5-4                                       //
//     Service               S-1-5-6                                       //
//     AnonymousLogon        S-1-5-7       (aka null logon session)        //
//                                                                         //
//     (Logon IDs)           S-1-5-5-X-Y                                   //
//                                                                         //
//     (NT non-unique IDs)   S-1-5-0x15-...                                //
//                                                                         //
//     (Built-in domain)     s-1-5-0x20                                    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////


SECURITY_NT_AUTHORITY: array[0..5] of Byte = (0,0,0,0,0,5);   // ntifs

SECURITY_DIALUP_RID: LongInt = $00000001;//     (0x00000001L)
SECURITY_NETWORK_RID: LongInt = $00000002;//    (0x00000002L)
SECURITY_BATCH_RID: LongInt = $00000003;//      (0x00000003L)
SECURITY_INTERACTIVE_RID: LongInt = $00000004;//   (0x00000004L)
SECURITY_SERVICE_RID: LongInt = $00000006;//    (0x00000006L)
SECURITY_ANONYMOUS_LOGON_RID: LongInt = $00000007;//    (0x00000007L)

SECURITY_LOGON_IDS_RID: LongInt = $00000005;//          (0x00000005L)
SECURITY_LOGON_IDS_RID_COUNT: LongInt = 3;//    (3L)

SECURITY_LOCAL_SYSTEM_RID: LongInt = $00000012;//       (0x00000012L)

SECURITY_NT_NON_UNIQUE: LongInt = $00000015;//          (0x00000015L)
SECURITY_BUILTIN_DOMAIN_RID: LongInt = $00000020;//     (0x00000020L)

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// well-known domain relative sub-authority values (RIDs)...               //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

// Well-known users ...

DOMAIN_USER_RID_ADMIN: LongInt = $000001F4;//          (0x000001F4L)
DOMAIN_USER_RID_GUEST: LongInt = $000001F5;//          (0x000001F5L)

// well-known groups ...

DOMAIN_GROUP_RID_ADMINS: LongInt = $00000200;//        (0x00000200L)
DOMAIN_GROUP_RID_USERS: LongInt = $00000201;//         (0x00000201L)
DOMAIN_GROUP_RID_GUESTS: LongInt = $00000202;//        (0x00000202L)

// well-known aliases ...

DOMAIN_ALIAS_RID_ADMINS: LongInt = $00000220;//        (0x00000220L)
DOMAIN_ALIAS_RID_USERS: LongInt = $00000221;//         (0x00000221L)
DOMAIN_ALIAS_RID_GUESTS: LongInt = $00000222;//        (0x00000222L)
DOMAIN_ALIAS_RID_POWER_USERS: LongInt = $00000223;//   (0x00000223L)

DOMAIN_ALIAS_RID_ACCOUNT_OPS: LongInt = $00000224;//   (0x00000224L)
DOMAIN_ALIAS_RID_SYSTEM_OPS: LongInt = $00000225;//    (0x00000225L)
DOMAIN_ALIAS_RID_PRINT_OPS: LongInt = $00000226;//     (0x00000226L)
DOMAIN_ALIAS_RID_BACKUP_OPS: LongInt = $00000227;//    (0x00000227L)

DOMAIN_ALIAS_RID_REPLICATOR: LongInt = $00000228;//    (0x00000228L)

////////////////////////////////////////////////////////////////////////
//                                                                    //
//                             SECURITY_DESCRIPTOR                    //
//                                                                    //
////////////////////////////////////////////////////////////////////////
//
//  Define the Security Descriptor and related data types.
//  This is an opaque data structure.
//

// begin_ntddk begin_ntifs
//
// Current security descriptor revision value
//

SECURITY_DESCRIPTOR_REVISION = 1;
SECURITY_DESCRIPTOR_REVISION1 = 1;

// end_ntddk

//
////////////////////////////////////////////////////////////////////////
//                                                                    //
//                         ACL  and  ACE                              //
//                                                                    //
////////////////////////////////////////////////////////////////////////

//
//  Define an ACL and the ACE format.  The structure of an ACL header
//  followed by one or more ACEs.  Pictorally the structure of an ACL header
//  is as follows:
//
//       3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
//       1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
//      +-------------------------------+---------------+---------------+
//      |            AclSize            |      Sbz1     |  AclRevision  |
//      +-------------------------------+---------------+---------------+
//      |              Sbz2             |           AceCount            |
//      +-------------------------------+-------------------------------+
//
//  The current AclRevision is defined to be ACL_REVISION.
//
//  AclSize is the size, in bytes, allocated for the ACL.  This includes
//  the ACL header, ACES, and remaining free space in the buffer.
//
//  AceCount is the number of ACES in the ACL.
//

// begin_ntddk begin_ntifs
// This is the *current* ACL revision

ACL_REVISION = 2;

// This is the history of ACL revisions.  Add a new one whenever
// ACL_REVISION is updated

ACL_REVISION1 = 1;
ACL_REVISION2 = 2;

//  The structure of an ACE is a common ace header followed by ace type
//  specific data.  Pictorally the structure of the common ace header is
//  as follows:
//
//       3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
//       1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
//      +---------------+-------+-------+---------------+---------------+
//      |            AceSize            |    AceFlags   |     AceType   |
//      +---------------+-------+-------+---------------+---------------+
//
//  AceType denotes the type of the ace, there are some predefined ace
//  types
//
//  AceSize is the size, in bytes, of ace.
//
//  AceFlags are the Ace flags for audit and inheritance, defined shortly.
Type
 ACE_HEADER = record
    AceType: BYTE;
    AceFlags: BYTE;
    AceSize: WORD;
 end;
 PACE_HEADER = ^ACE_HEADER;
//
//  The following are the predefined ace types that go into the AceType
//  field of an Ace header.
//
Const
ACCESS_ALLOWED_ACE_TYPE =        $0;
ACCESS_DENIED_ACE_TYPE = $1;
SYSTEM_AUDIT_ACE_TYPE  = $2;
SYSTEM_ALARM_ACE_TYPE  = $3;

//
//  The following are the inherit flags that go into the AceFlags field
//  of an Ace header.
//

OBJECT_INHERIT_ACE = $1;
CONTAINER_INHERIT_ACE  =    $2;
NO_PROPAGATE_INHERIT_ACE =  $4;
INHERIT_ONLY_ACE         =  $8;
VALID_INHERIT_FLAGS      =  $F;


//  The following are the currently defined ACE flags that go into the
//  AceFlags field of an ACE header.  Each ACE type has its own set of
//  AceFlags.
//
//  SUCCESSFUL_ACCESS_ACE_FLAG - used only with system audit and alarm ACE
//  types to indicate that a message is generated for successful accesses.
//
//  FAILED_ACCESS_ACE_FLAG - used only with system audit and alarm ACE types
//  to indicate that a message is generated for failed accesses.
//

//
//  SYSTEM_AUDIT and SYSTEM_ALARM AceFlags
//
//  These control the signaling of audit and alarms for success or failure.
//

SUCCESSFUL_ACCESS_ACE_FLAG =      $40;
FAILED_ACCESS_ACE_FLAG   =        $80;


//
//  We'll define the structure of the predefined ACE types.  Pictorally
//  the structure of the predefined ACE's is as follows:
//
//       3 3 2 2 2 2 2 2 2 2 2 2 1 1 1 1 1 1 1 1 1 1
//       1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
//      +---------------+-------+-------+---------------+---------------+
//      |    AceFlags   | Resd  |Inherit|    AceSize    |     AceType   |
//      +---------------+-------+-------+---------------+---------------+
//      |                              Mask                             |
//      +---------------------------------------------------------------+
//      |                                                               |
//      +                                                               +
//      |                                                               |
//      +                              Sid                              +
//      |                                                               |
//      +                                                               +
//      |                                                               |
//      +---------------------------------------------------------------+
//
//  Mask is the access mask associated with the ACE.  This is either the
//  access allowed, access denied, audit, or alarm mask.
//
//  Sid is the Sid associated with the ACE.
//

//  The following are the four predefined ACE types.

//  Examine the AceType field in the Header to determine
//  which structure is appropriate to use for casting.

Type

ACCESS_ALLOWED_ACE = record
    Header: ACE_HEADER;
    Mask: ACCESS_MASK;
    SidStart: DWORD;
end;

PACCESS_ALLOWED_ACE = ^ACCESS_ALLOWED_ACE;

ACCESS_DENIED_ACE = record
    Header: ACE_HEADER;
    Mask: ACCESS_MASK;
    SidStart: DWORD;
end;

PACCESS_DENIED_ACE = ^ACCESS_DENIED_ACE;

SYSTEM_AUDIT_ACE = record
    Header: ACE_HEADER;
    Mask: ACCESS_MASK;
    SidStart: DWORD;
end;

PSYSTEM_AUDIT_ACE = ^SYSTEM_AUDIT_ACE;

SYSTEM_ALARM_ACE = record
    Header: ACE_HEADER;
    Mask: ACCESS_MASK;
    SidStart: DWORD;
end;
PSYSTEM_ALARM_ACE = ^SYSTEM_ALARM_ACE;

// this initializes the application for security manipulation
Type

ESecObjError = class(Exception)
Private
 FErrorCode: Integer;
 Function GetLastErrorText: String;
Public
 Constructor Create(Msg: String; Error: Integer);virtual;
 Property ErrorCode: Integer Read FErrorCode Write FErrorCode;
end;

TSecureableObject = class(TObject)
 protected
  m_pSD: PSecurityDescriptor;
  m_pDACL: PACL;
  m_pSACL: PACL;
  m_pOwner: PSID;
  m_pPrimaryGroup: PSID;
  function GetSIDFromName(pDomainName, pAccountName: String; var pcSid: PSID; var pcDomainName: PChar): Boolean;
 public
  m_iSecErrorCode: integer;
 protected
  procedure SetTheDescriptor;virtual;abstract;
  procedure GetTheDescriptor;virtual;abstract;
  procedure BuildSD(pSelfRelativeReturnSD: PSecurityDescriptor);
  procedure FreeDataStructures;
  procedure ZeroOut;
 public
  constructor Create (bProtected: Boolean); virtual;
  destructor Destroy; virtual;
  function AddRightsTo(pDomainName, pAccountName: String; dwAccessMask: DWORD; bGranted: Boolean): Boolean;
  function RevokePreviouslyGrantedAccess(pAccountName, pDomainName: String): Boolean;
end;

TKernelSecObject = class(TSecureableObject)
public
  procedure SetObjectSecurity(hObject: THandle);
  procedure GetObjectSecurity(hObject: THandle);
end;

TUserSecObject = class(TSecureableObject)
public
  procedure SetObjectSecurity(hObject: THandle);
  procedure GetObjectSecurity(hObject: THandle);
end;
{
TServiceSecObject = class(TSecureableObject)
public
  procedure SetObjectSecurity(hObject: TSC_Handle);
  procedure GetObjectSecurity(hObject: TSC_Handle);
end;
}

{TPrivateSecObject = class(TSecureableObject)
private
  m_hAccessToken: HANDLE;
  m_gmPrivateMapping: TGeneric_Mapping;
  m_ObjectSD: PSecurityDescriptor;
  m_dwUniqueHandleId: Pointer;
public
  Constructor Create(bProtected: Boolean); override;
  Destructor Destroy; override;
  procedure SetObjectSecurity;
  procedure GetObjectSecurity;
  procedure MatchAccessRequest(dwMask: DWORD; hThread: THandle);
  function MapRequestedRights (dwArg: DWORD): DWORD;
end;
}
function SetSpecificPrivilegeInAccessToken(lpPrivType: PAnsiChar; bEnabled: Boolean): Boolean;
function SetPrivilegeInAccessToken(bEnabled: Boolean): Boolean;

implementation

{ ESecObjError }
Constructor ESecObjError.Create(Msg: String; Error: Integer);
begin
 FErrorCode := Error;
 Msg := Msg + ' ' + GetLastErrorText;
 inherited Create(Msg);
end;

Function ESecObjError.GetLastErrorText: String;
Var
  dwRet: Integer;
  szTemp: Array[0..511] Of Char;
Begin
  Result:= '';
  dwRet:= FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM Or FORMAT_MESSAGE_ARGUMENT_ARRAY,
                        Nil, FErrorCode, LANG_NEUTRAL,
                        szTemp, SizeOf(szTemp)-2, Nil);

  szTemp[StrLen(szTemp)-2]:= #0;  //remove cr and newline character
  Result:= Format('%s (0x%x)', [szTemp, GetLastError])
End;

// helper function
function SetSpecificPrivilegeInAccessToken(lpPrivType: PAnsiChar; bEnabled: Boolean): Boolean;
var
  hProcess:        THandle;
  hAccessToken:    THandle;
  luidPrivilegeLUID: TLargeInteger;
  tpTokenPrivilege, prevTP: TTokenPrivileges;
  ReturnLength: DWORD;
begin
  Result := False;

  hProcess := GetCurrentProcess;
  if hProcess = 0 then
     exit;

  if Not OpenProcessToken(hProcess,
                        TOKEN_ADJUST_PRIVILEGES Or TOKEN_QUERY,
                        @hAccessToken) then
     exit;
 TRY
  if Not LookupPrivilegeValue(nil,
                            lpPrivType,
                            luidPrivilegeLUID) then
     exit;

  tpTokenPrivilege.PrivilegeCount := 1;
  tpTokenPrivilege.Privileges[0].Luid := luidPrivilegeLUID;
  if bEnabled then
     tpTokenPrivilege.Privileges[0].Attributes := SE_PRIVILEGE_ENABLED
  else
     tpTokenPrivilege.Privileges[0].Attributes := 0;
  SetLastError(ERROR_SUCCESS);

  Result := AdjustTokenPrivileges (hAccessToken,
                         FALSE,  // Do not disable all
                         tpTokenPrivilege,
                         sizeof(TTokenPrivileges),
                         prevTP,   // Ignore previous info
                         ReturnLength);  // Ignore previous info
  if (GetLastError <> ERROR_SUCCESS) then Result := FALSE;
 FINALLY
  CloseHandle(hAccessToken);
 END;
end;

function SetPrivilegeInAccessToken(bEnabled: Boolean): Boolean;
begin
 Result := SetSpecificPrivilegeInAccessToken('SeSecurityPrivilege',bEnabled);
end;


Constructor TSecureableObject.Create(bProtected: Boolean);
var
 dwDACLLength: DWORD;
 pcSid: PSID;
 siaWorld: TSIDIdentifierAuthority;
 iTempSidLength: integer;
begin
 ZeroOut; pcSid := nil;
 Move(SECURITY_WORLD_SID_AUTHORITY, siaWorld.Value, SizeOf(SECURITY_WORLD_SID_AUTHORITY));
 GetMem(m_pSD, sizeof(TSecurityDescriptor));
TRY {except}
 if (m_pSD = nil) then
  raise ESecObjError.Create('Not Enough Memory', ERROR_NOT_ENOUGH_MEMORY);

 if Not InitializeSecurityDescriptor(m_pSD,SECURITY_DESCRIPTOR_REVISION) then
  raise ESecObjError.Create('InitializeSecurityDS failed', GetLastError);

 iTempSidLength := GetSidLengthRequired(1); // this can not fail
 GetMem(pcSid, iTempSidLength);
 if (pcSid = nil) then
  raise ESecObjError.Create('Not Enough Memory', ERROR_NOT_ENOUGH_MEMORY);
TRY{finally}
 dwDACLLength := sizeof(TACL) + sizeof(ACCESS_ALLOWED_ACE) - sizeof(DWORD) + iTempSidLength;
 GetMem(m_pDACL, dwDACLLength);
 if (m_pDACL = nil) then
  raise ESecObjError.Create('Not Enough Memory', ERROR_NOT_ENOUGH_MEMORY);

 if (Not InitializeAcl(m_pDACL^,dwDACLLength,ACL_REVISION)) Or (Not InitializeSid(pcSid,siaWorld,1)) then
  raise ESecObjError.Create('Initialization of Security object failed', GetLastError);

 (GetSidSubAuthority(pcSid,0))^ := SECURITY_WORLD_RID;
 if (bProtected) then begin // this works like a empty dacl - fully protected
   if (Not AddAccessAllowedAce(m_pDACL^,ACL_REVISION,NULL,pcSid)) then // RAc
      raise ESecObjError.Create('AddAccessAllowedAce failed', GetLastError);
 end
 else begin               // this works like a NULL dacl - unprotected
  if (Not AddAccessAllowedAce(m_pDACL^,ACL_REVISION,GENERIC_ALL Or STANDARD_RIGHTS_ALL Or SPECIFIC_RIGHTS_ALL,pcSid)) then
     raise ESecObjError.Create('AddAccessAllowedAce failed', GetLastError);
 end;

 if ( Not SetSecurityDescriptorDacl(m_pSD,TRUE,m_pDACL,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorDacl failed', GetLastError);

 // at this point we need to go through the same procedure w/ a SACL...
 if (pcSid <> nil) then FreeMem(pcSid);
 pcSid := nil;             // we reuse this data structure.
 iTempSidLength := GetSidLengthRequired(1); // this can not fail
 GetMem(pcSid, iTempSidLength);
 if (pcSid = nil) then
     raise ESecObjError.Create('Not Enough Memory', ERROR_NOT_ENOUGH_MEMORY);
 dwDACLLength := sizeof(TACL) + sizeof(SYSTEM_AUDIT_ACE) - sizeof(DWORD) + iTempSidLength;
 GetMem(m_pSACL, dwDACLLength);
 if (m_pSACL = nil) then
    raise ESecObjError.Create('Not Enough Memory', ERROR_NOT_ENOUGH_MEMORY);

 if ( Not InitializeAcl(m_pSACL^,dwDACLLength,ACL_REVISION)) Or (Not InitializeSid(pcSid,siaWorld,1)) then
    raise ESecObjError.Create('AddAccessAllowedAce failed', GetLastError);
 (GetSidSubAuthority(pcSid,0))^ := SECURITY_WORLD_RID;
 if (Not AddAuditAccessAce(m_pSACL^,ACL_REVISION,GENERIC_ALL Or STANDARD_RIGHTS_ALL Or SPECIFIC_RIGHTS_ALL,pcSid,TRUE,TRUE)) then
    raise ESecObjError.Create('AddAuditAllowedAce failed', GetLastError);
 if (Not SetSecurityDescriptorSacl(m_pSD,TRUE,m_pSACL,FALSE)) Then
    raise ESecObjError.Create('SetSecurityDescriptorSacl failed', GetLastError);

 SetLastError(ERROR_SUCCESS);
FINALLY
 if pcSid <> nil then FreeMem(pcSid);
END;
EXCEPT
 on E:ESecObjError do begin
   m_iSecErrorCode := E.ErrorCode;
   raise;
 end;
END;
end;

Destructor TSecureableObject.Destroy;
begin
 FreeDataStructures;
end;

procedure TSecureableObject.FreeDataStructures;
begin
   if (m_pSD <> nil) then FreeMem(m_pSD);
   if (m_pDACL <> nil) then FreeMem(m_pDACL);
   if (m_pSACL <> nil) then FreeMem(m_pSACL);
   if (m_pOwner <> nil) then FreeMem(m_pOwner);
   if (m_pPrimaryGroup <> nil) then FreeMem(m_pPrimaryGroup);
   ZeroOut;
end;

procedure TSecureableObject.ZeroOut;
begin
   m_pSD := nil;
   m_pDACL := nil;
   m_pSACL := nil;
   m_pOwner := nil;
   m_pPrimaryGroup := nil;
end;

function TSecureableObject.AddRightsTo(pDomainName, pAccountName: String;dwAccessMask: DWORD;bGranted: Boolean): Boolean;
var
 dwDACLLength: DWORD;
 pcDomainName: PChar;
 pcSid: PSID;
 bHasDacl,bHasDefaulted: BOOL;
 pCurrentAcl: PACL;
 pNewACL: PACL;
 pNewSD: PSecurityDescriptor;
 dwNumberOfAces: Integer;
 dwAceSize, dwCurrentSecurityDescriptorLength: DWORD;
 iLoop: integer;
 pTempACL: Pointer;

begin
pCurrentAcl := nil; pNewACL := nil; pNewSD := nil;
TRY {except}
 if (Not GetSIDFromName(pDomainName,pAccountName, pcSid, pcDomainName)) then
  raise ESecObjError.Create('GetSIDFromName failed', m_iSecErrorCode);

 GetTheDescriptor; // this will implicitly set m_pSD

 if (Not GetSecurityDescriptorDacl(m_pSD, bHasDacl, pCurrentAcl, bHasDefaulted)) then
   raise ESecObjError.Create('GetSecurityDescriptorDacl failed', GetLastError);

 dwNumberOfAces := pCurrentAcl^.AceCount;
 dwAceSize := pCurrentAcl^.AclSize;
 dwDACLLength := dwAceSize + sizeof(ACCESS_ALLOWED_ACE) - sizeof(DWORD) +
	         GetLengthSid(pcSid);		// we must assume here that the sid is valid...
 GetMem(pNewACL, dwDACLLength);
 if (pNewACL = nil) then
  raise ESecObjError.Create('Memory Allocation Failed', ERROR_NOT_ENOUGH_MEMORY);
 dwCurrentSecurityDescriptorLength := GetSecurityDescriptorLength(m_pSD);
 GetMem(pNewSD, dwDACLLength+dwCurrentSecurityDescriptorLength);
 if (pNewSD = nil) then
    raise ESecObjError.Create('Memory Allocation Failed', ERROR_NOT_ENOUGH_MEMORY);

 if (Not InitializeSecurityDescriptor(pNewSD,SECURITY_DESCRIPTOR_REVISION)) then
    raise ESecObjError.Create('InitializeSecurityDescriptor Failed', GetLastError);
 if (Not InitializeAcl(pNewACL^,dwDACLLength,ACL_REVISION)) then
    raise ESecObjError.Create('InitializeAcl Failed', GetLastError);

// beginning of specific code -- depending on whether we want to grant or deny
// access, we must either add an access allowed ace at the end, or tuck a denied ace up front...
 if ( Not bGranted) then
  if (Not AddAccessDeniedAce(pNewACL^,ACL_REVISION,dwAccessMask,PSID(pcSid))) then
     raise ESecObjError.Create('AddAccessDeniedAce Failed', GetLastError);

 for iLoop := 0 to ( dwNumberOfAces - 1) do begin
   if (Not GetAce(pCurrentAcl^, iLoop, pTempACL)) then
     raise ESecObjError.Create('GetAce Failed', GetLastError);
   if (Not AddAce(pNewACL^,ACL_REVISION,MAXDWORD,pTempACL,PACE_HEADER(pTempACL)^.AceSize)) then
     raise ESecObjError.Create('AddAce Failed', GetLastError);
 end;{for}

 if bGranted then
  if ( Not AddAccessAllowedAce(pNewACL^,ACL_REVISION,dwAccessMask,PSID(pcSid))) then
     raise ESecObjError.Create('AddAccessAllowedAce Failed', GetLastError);

// now we may nuke the old SD because we don't need it anymore.
// note that we keep it until we know that security was set correctly. This way,
//if anything fails, we'll still have the old one around...
 if ( Not IsValidAcl(pNewACL^))then
    raise ESecObjError.Create('Invalid ACL', GetLastError);
 if (Not IsValidSecurityDescriptor(pNewSD)) then
    raise ESecObjError.Create('Invalid SD', GetLastError);
 if (Not SetSecurityDescriptorDacl(pNewSD,TRUE,pNewACL,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorDacl Failed', GetLastError);
// now copy all of the other relevant stuff to the new SD
 if (Assigned(m_pOwner)) And (Not SetSecurityDescriptorOwner(pNewSD,m_pOwner,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorOwner Failed', GetLastError);
 if (Assigned(m_pSACL)) And (Not SetSecurityDescriptorSacl(pNewSD,TRUE,m_pSACL,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorSacl Failed', GetLastError);
 if (Assigned(m_pPrimaryGroup)) And (Not SetSecurityDescriptorGroup(pNewSD,m_pPrimaryGroup,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorGroup Failed', GetLastError);

 If Assigned(m_pSD) then FreeMem(m_pSD);
 m_pSD := pNewSD;
 if Assigned(m_pDACL) then FreeMem(m_pDACL);
 m_pDACL := pNewACL;
// If Assigned(pCurrentAcl) then FreeMem(pCurrentAcl);
 SetTheDescriptor;
 Result := True;
EXCEPT
 Result := False;
// if Assigned(pNewACL) Then FreeMem(pNewACL);
// if Assigned(pNewSD) Then FreeMem(pNewSD);
 raise;
END;
end;


function TSecureableObject.RevokePreviouslyGrantedAccess(pAccountName, pDomainName: String): Boolean;
var
 pNewACL: PACL;
 pCurrentAcl: PACL;
 dwDACLLength: DWORD;
 pcDomainName: PChar;
 pcSid: PSID;
 pNewSD: PSecurityDescriptor;
 iOffendingIndex: integer;
 bHasDacl: BOOL;
 bHasDefaulted: BOOL;
 dwNumberOfAces, iLoop: Integer;
 dwAceSize, dwCurrentSecurityDescriptorLength: DWORD;
 pTempACL: Pointer;
 iFound: Boolean;
begin
 pNewACL := nil; pCurrentAcl := nil; pNewSD := nil;
TRY
 if (Not GetSIDFromName(pDomainName,pAccountName, pcSid, pcDomainName)) then
    raise ESecObjError.Create('GetSIDFromName Failed', m_iSecErrorCode);

 GetTheDescriptor;// this will implicitly set m_pSD
 if (Not GetSecurityDescriptorDacl(m_pSD, bHasDacl, PACL(pCurrentAcl), bHasDefaulted)) then
    raise ESecObjError.Create('GetTheDescriptor Failed', GetLastError);
 dwNumberOfAces := pCurrentAcl^.AceCount;
 dwAceSize := pCurrentAcl^.AclSize;

 dwCurrentSecurityDescriptorLength := GetSecurityDescriptorLength(m_pSD);
// here we figure out if an ace with the requested sid is already in the acl...
 iFound := False;
 for iLoop := 0 to (dwNumberOfAces - 1) do begin
  if (Not GetAce(pCurrentAcl^, iLoop, pTempACL)) then
     raise ESecObjError.Create('GetAce Failed', GetLastError);
  iOffendingIndex := iLoop;
  iFound := True;
  if (EqualSid (PSID( @PACCESS_ALLOWED_ACE(pTempACL)^.SidStart),pcSid)) then
      break;
  iFound := False;
 end;

 if Not iFound then
{ iLoop >= dwNumberOfAces)}
//  SetLastError(ERROR_PRIVILEGE_NOT_HELD);
//  goto ErrorExit;    // didn't find the sid...
  raise ESecObjError.Create('GetAce Failed', ERROR_PRIVILEGE_NOT_HELD);


 dwDACLLength := dwAceSize - PACE_HEADER(pTempACL)^.AceSize;
 GetMem(pNewACL,dwDACLLength);
 if (pNewACL = nil) then
    raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
 GetMem(pNewSD,(dwDACLLength + dwCurrentSecurityDescriptorLength));
 if (pNewSD = nil) then
    raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
 if (Not InitializeSecurityDescriptor(pNewSD,SECURITY_DESCRIPTOR_REVISION)) then
    raise ESecObjError.Create('InitializeSecurityDescriptor Failed', GetLastError);
 if ( Not InitializeAcl(pNewACL^,dwDACLLength,ACL_REVISION)) then
    raise ESecObjError.Create('InitializeAcl Failed', GetLastError);
// now we copy the old ace to the new one, of course leaving out the offending sid...
 for iLoop := 0 to (dwNumberOfAces -1) do begin
  if (iLoop = iOffendingIndex) then continue;
  if (Not GetAce(pCurrentAcl^, iLoop, pTempACL)) then
     raise ESecObjError.Create('GetAce Failed', GetLastError);
  if (Not AddAce(pNewACL^,ACL_REVISION,MAXDWORD,pTempACL, PACE_HEADER(pTempACL)^.AceSize)) then
     raise ESecObjError.Create('AddAce Failed', GetLastError);
 end;{for}
 // now we may nuke the old ACL because we don't need it anymore...
 if (Not IsValidAcl(pNewACL^)) then
    raise ESecObjError.Create('IsValidAcl Failed', GetLastError);
 if (Not IsValidSecurityDescriptor(pNewSD)) then
    raise ESecObjError.Create('IsValidSecurityDescriptor Failed', GetLastError);
 if (Not SetSecurityDescriptorDacl(pNewSD,TRUE,pNewACL,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorDacl Failed', GetLastError);
// now copy all of the other relevant stuff to the new SD
 if (m_pOwner <> nil) And (Not SetSecurityDescriptorOwner(pNewSD,m_pOwner,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorOwner Failed', GetLastError);
 if (m_pSACL <> nil) And (Not SetSecurityDescriptorSacl(pNewSD,TRUE,m_pSACL,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorSacl Failed', GetLastError);
 if (m_pPrimaryGroup <> nil) And (Not SetSecurityDescriptorGroup(pNewSD,m_pPrimaryGroup,FALSE)) then
    raise ESecObjError.Create('SetSecurityDescriptorGroup Failed', GetLastError);
 FreeMem(pCurrentAcl);
 if Assigned (m_pSD) then FreeMem(m_pSD);
 m_pSD := pNewSD;
// if Assigned(m_pDACL) then FreeMem(m_pDACL);
 m_pDACL := pNewACL;
 SetTheDescriptor;
// free (pACL);
EXCEPT
// if Assigned(pNewACL) then FreeMem(pNewACL);
// if Assigned(pNewSD) then FreeMem(pNewSD);
 raise;
END;
end;


// helper functions

function TSecureableObject.GetSIDFromName(pDomainName, pAccountName: String; var pcSid: PSID; var pcDomainName: PChar): Boolean;
var
 su: SID_NAME_USE;
 dwSidLength: DWORD;
 iBufLen: DWORD;
begin
 dwSidLength := 0; iBufLen := 0;
 (* we first must determine the SID that belongs to the user. This happens in
    two steps: First, determine how much space is needed for the SID and
	domain buffers. Then, call the function again with the previously allocated
	buffers. *)
 Result := False;

 if (Not LookupAccountName(PChar(pDomainName), PChar(pAccountName),pcSid,dwSidLength,pcDomainName,
       iBufLen,su)) then begin
// this call will fail in any case, or we are in trouble.
  if (GetLastError <> ERROR_INSUFFICIENT_BUFFER) then
// didn't find the name, or something else went wrong...
     raise ESecObjError.Create('LookupAccountName Failed', GetLastError);
  GetMem(pcSid, dwSidLength);
  if (pcSid = nil) then
     raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  GetMem(pcDomainName, iBufLen);
  if (pcDomainName = nil) then begin
     FreeMem(pcSid);
     raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  end;
  if (Not LookupAccountName(PChar(pDomainName),PChar(pAccountName),pcSid,dwSidLength,pcDomainName,
        iBufLen,su)) then begin
   FreeMem(pcSid);
   FreeMem(pcDomainName);
   raise ESecObjError.Create('LookupAccountName Failed', GetLastError);
  end;
  if (Not IsValidSid(pcSid)) then begin
   FreeMem(pcSid);
   FreeMem(pcDomainName);
   raise ESecObjError.Create('IsValidSid Failed', GetLastError);
  end;
 end;
 Result := TRUE;
end;

procedure TSecureableObject.BuildSD(pSelfRelativeReturnSD: PSecurityDescriptor);
var
 iSDSize,iDACLSize,iSACLSize,iOwnerSize,iGroupSize: DWORD;
begin
 iSDSize:=0;iDACLSize:=0;iSACLSize:=0;iOwnerSize:=0;iGroupSize:=0;
TRY
 if (Not MakeAbsoluteSD(pSelfRelativeReturnSD,m_pSD,iSDSize, m_pDACL^,iDACLSize,
                        m_pSACL^,iSACLSize, m_pOwner,iOwnerSize, m_pPrimaryGroup,iGroupSize)) And
	 (GetLastError = ERROR_INSUFFICIENT_BUFFER) then begin
  // this is tedious: Allocate all the data structures and try again...
  if (iSDSize > 0) then begin
   GetMem(m_pSD,iSDSize);
   if (m_pSD = nil) then
      raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  end;
  if (iDACLSize > 0) then begin
   GetMem(m_pDACL,iDACLSize);
   if (m_pDACL = nil) then
      raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  end;
  if (iSACLSize > 0) then begin
   GetMem(m_pSACL,iSACLSize);
   if (m_pSACL = nil) then
      raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  end;
  if (iOwnerSize > 0) then begin
   GetMem(m_pOwner,iOwnerSize);
   if (m_pOwner = nil) then
      raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  end;
  if (iGroupSize > 0) then begin
   GetMem(m_pPrimaryGroup,iGroupSize);
   if (m_pPrimaryGroup = nil) then
      raise ESecObjError.Create('Low Memory', ERROR_NOT_ENOUGH_MEMORY);
  end; // end of allocation stuff

  if (Not MakeAbsoluteSD(pSelfRelativeReturnSD, m_pSD,iSDSize,m_pDACL^,iDACLSize,
			 m_pSACL^,iSACLSize,m_pOwner,iOwnerSize,m_pPrimaryGroup,iGroupSize)) then
     raise ESecObjError.Create('MakeAbsoluteSD Failed', GetLastError);
  // by now we should have a good SD in m_pSD...
 end
  else
   raise ESecObjError.Create('MakeAbsoluteSD Failed', GetLastError);
EXCEPT
 FreeDataStructures;
 raise;
END;
end;

// derived classes...

{TKernelSecObject}
procedure TKernelSecObject.SetObjectSecurity(hObject: THandle);
begin
TRY
 if (Not SetKernelObjectSecurity(hObject,(DACL_SECURITY_INFORMATION{ Or SACL_SECURITY_INFORMATION}),m_pSD)) then
  raise ESecObjError.Create('SetKernelObjectSecurity Failed', GetLastError);
FINALLY
 FreeDataStructures;
END;
end;

procedure TKernelSecObject.GetObjectSecurity(hObject: THandle);
var
 pSelfRelativeReturnSD: PSecurityDescriptor;
 iSDSize: DWORD;
begin
 iSDSize := 0; pSelfRelativeReturnSD := nil;
 if (Not GetKernelObjectSecurity(hObject,DACL_SECURITY_INFORMATION{ Or SACL_SECURITY_INFORMATION},pSelfRelativeReturnSD,iSDSize,iSDSize)) AND
      (GetLastError = ERROR_INSUFFICIENT_BUFFER) then begin
  GetMem(pSelfRelativeReturnSD, iSDSize);
 TRY
  if (pSelfRelativeReturnSD = nil) then
    raise ESecObjError.Create('GetKernelObjectSecurity Failed', ERROR_NOT_ENOUGH_MEMORY);
  if ( Not GetKernelObjectSecurity(hObject,DACL_SECURITY_INFORMATION{ Or SACL_SECURITY_INFORMATION},pSelfRelativeReturnSD,iSDSize,iSDSize)) then
    raise ESecObjError.Create('GetKernelObjectSecurity Failed', GetLastError);
  BuildSD(pSelfRelativeReturnSD);
 FINALLY
  if pSelfRelativeReturnSD <> nil then FreeMem(pSelfRelativeReturnSD);
 END;
 end;
end;

procedure TUserSecObject.SetObjectSecurity(hObject: THandle);
var
 si: SECURITY_INFORMATION;
begin
si := DACL_SECURITY_INFORMATION;// Or SACL_SECURITY_INFORMATION;
TRY
 if (Not SetUserObjectSecurity(hObject,si,m_pSD)) then
    raise ESecObjError.Create('SetUserObjectSecurity Failed', GetLastError);
FINALLY
 FreeDataStructures;
END;
end;

procedure TUserSecObject.GetObjectSecurity(hObject: THandle);
var
 pSelfRelativeReturnSD: PSecurityDescriptor;
 iSDSize: DWORD;
 si: SECURITY_INFORMATION;
begin
 pSelfRelativeReturnSD := nil; iSDSize := 0; si := DACL_SECURITY_INFORMATION;// Or SACL_SECURITY_INFORMATION;
 if (Not GetUserObjectSecurity(hObject, si,pSelfRelativeReturnSD,iSDSize,iSDSize)) AND
      (GetLastError = ERROR_INSUFFICIENT_BUFFER) then begin
  GetMem(pSelfRelativeReturnSD, iSDSize);
 TRY
  if (pSelfRelativeReturnSD = nil) then
     raise ESecObjError.Create('GetUserObjectSecurity Failed', ERROR_NOT_ENOUGH_MEMORY);
  if (Not GetUserObjectSecurity(hObject,si,pSelfRelativeReturnSD,iSDSize,iSDSize)) Then
     raise ESecObjError.Create('GetUserObjectSecurity Failed', GetLastError);
  BuildSD(pSelfRelativeReturnSD);
 FINALLY
  if pSelfRelativeReturnSD <> nil then FreeMem(pSelfRelativeReturnSD);
 END;
 end;
end;


{Constructor TPrivateSecObject.Create(bProtected: Boolean);
var
 hProcess: THandle;
begin
 hProcess := GetCurrentProcess;
 if( Not OpenProcessToken(hProcess,TOKEN_ALL_ACCESS, m_hAccessToken)) then
   m_iSecErrorCode =GetLastError;
 SetPrivilegeInAccessToken(TRUE);
 m_gmPrivateMapping.GenericRead = DBASE_READ;
 m_gmPrivateMapping.GenericWrite = DBASE_WRITE;
 m_gmPrivateMapping.GenericExecute = STANDARD_RIGHTS_EXECUTE;
 m_gmPrivateMapping.GenericAll = DBASE_READ|DBASE_WRITE;
 if (Not CreatePrivateObjectSecurity(NULL,m_pSD,m_ObjectSD,FALSE,m_hAccessToken,m_gmPrivateMapping)) then
  m_iSecErrorCode = GetLastError;
 SetPrivilegeInAccessToken(FALSE);
 m_dwUniqueHandleId = Self;
end;

//CPrivateSecObject::~CPrivateSecObject()
{
 CloseHandle(m_hAccessToken);
 DestroyPrivateObjectSecurity(&m_ObjectSD);
}//;

//BOOL CPrivateSecObject::SetObjectSecurity()
{
 BOOL bErrorCode = TRUE;
 if (!SetPrivateObjectSecurity(DACL_SECURITY_INFORMATION|SACL_SECURITY_INFORMATION,
       m_pSD,&m_ObjectSD,&m_gmPrivateMapping,m_hAccessToken))
 {
  m_iSecErrorCode = GetLastError();
  bErrorCode = FALSE;
 }//;
// FreeDataStructures();
// return bErrorCode;
//};

//BOOL CPrivateSecObject::GetObjectSecurity()
{
 PSECURITY_DESCRIPTOR pSelfRelativeReturnSD=NULL;
 DWORD iSDSize=0;
 if (!GetPrivateObjectSecurity(m_ObjectSD,DACL_SECURITY_INFORMATION|SACL_SECURITY_INFORMATION,pSelfRelativeReturnSD,iSDSize,&iSDSize) &&
      GetLastError() == ERROR_INSUFFICIENT_BUFFER)
 {
  pSelfRelativeReturnSD = malloc(iSDSize);
  if (!pSelfRelativeReturnSD)
  {
   SetLastError(ERROR_NOT_ENOUGH_MEMORY);
   goto ErrorExit;
  }//;
 // if (!GetPrivateObjectSecurity(m_ObjectSD,DACL_SECURITY_INFORMATION|SACL_SECURITY_INFORMATION,pSelfRelativeReturnSD,iSDSize,&iSDSize))
//   goto ErrorExit;
//  !BuildSD(pSelfRelativeReturnSD);
//  free (pSelfRelativeReturnSD);
//  return TRUE;
// };
//ErrorExit:
// if (pSelfRelativeReturnSD) free (pSelfRelativeReturnSD);
// return FALSE;
//};

//  procedure MatchAccessRequest(dwMask: DWORD; hThread: THandle);
//  function MapRequestedRights (dwArg: DWORD): DWORD;

//BOOL CPrivateSecObject::MatchAccessRequest(DWORD dwMask, HANDLE hThread)
{
 BOOL bReturn;
 DWORD dwStatus;
 BOOL bAuditGenerated;
 if(!AccessCheckAndAuditAlarm("DBServer",(LPVOID)m_dwUniqueHandleId,"Database Object","TestBase",
                              m_ObjectSD,dwMask,&m_gmPrivateMapping,TRUE,
							  &dwStatus,&bReturn,&bAuditGenerated))
{
  m_iSecErrorCode = GetLastError();
  return FALSE;
 }//;
// if (!ObjectCloseAuditAlarm("DBServer",(LPVOID)m_dwUniqueHandleId,bAuditGenerated))
 {
  m_iSecErrorCode = GetLastError();
  return FALSE;
 }//;
// m_iSecErrorCode = ERROR_SUCCESS;
// m_dwUniqueHandleId++;
// return bReturn;
//};

//DWORD CPrivateSecObject::MapRequestedRights (DWORD dwArg)
{
 MapGenericMask(&dwArg,&m_gmPrivateMapping);
 return dwArg;
}//;


end.
