{ Unit:      SvcObj
  Version:   1.07
  Purpose:   Components around winsvc.pas

 Author:     Peter Sawatzki (ps)
             Buchenhof 3, 58091 Hagen, Germany
 CompuServe: 100031,3002

  Date:    Author:
  03/01/96 ps     first version for Delphi 2.0
  04/02/96 ps     use thunks to make multiple service instances possible
  04/10/96 ps     add proper event logging
  05/01/96 ps     add pause/continue,
                  add watchdog to terminate non-responding service
  05/03/96 ps     add Interactive, Account, Password

  Copyright (c) 1996 Peter Sawatzki. All Rights Reserved.

}

Unit SvcObj;
Interface
Uses
  SysUtils,
  Classes,
  Windows,
  WinSvc,
  WinNT,
  NTObj;

Type
  EService = Class(Exception);

  TServiceType = (stWin32, stKernel, stFileSystem);
  TService = Class(TThread)
  Private
  {fields in API format}
    FDependencies: Array[0..100] Of Char;
    FStartType: DWord;
    FStatus: TService_Status;
    FStatusHandle: TService_Status_Handle;

  {private fields}
    FAccount: String;
    FCanPause: Boolean;
    FCanHardStop: Boolean;
    FDebug: Boolean;
    FDisplayName: String;
    FInteractive: Boolean;
    FPassword: String;
    FServiceName: String;
    FServiceType: TServiceType;
    FWaitHint: Integer;

    FMainWrapper, FCtrlWrapper: TWrapper;

    Function GetDependencies: String;
    Function GetStatus: Integer;
    Function GetStatusHandle: TService_Status_Handle;
    Procedure SetServiceName (Const Value: String);
    Procedure SetDependencies (Const Value: String);
    Procedure SetStatus (Value: Integer);
  Protected
    StopEvent: TEvent;
    EventLog: TEventLog;
    Procedure Ctrl (Code: Integer); StdCall;
    Procedure Main (NumArgs: DWord; Var Args: PChar); StdCall;

    Procedure Start; Virtual;
    Procedure Execute; Override;
    Procedure Stop; Virtual;
    Procedure InternalPause;
    Procedure InternalContinue;
    Procedure InternalStop;
    Procedure InitStatus; Virtual;
    Function ProcessType: Integer;
    Property StatusHandle: TService_Status_Handle Read GetStatusHandle;
    Property Status: Integer Read GetStatus Write SetStatus;
    Property Debug: Boolean Read FDebug Write FDebug;
  Public
    Constructor Create (Const AServiceName, ADisplayName: String);
    Destructor Destroy; Override;
    Function ServiceEntry: TService_Table_Entry;
    Procedure Check (Value: Bool);
    Procedure AddToEventLog (AType: TEventLogType; Value: String);
    Function GetLastErrorText: String;
    Procedure ReportStatus;
    Procedure Install;
    Procedure Remove;
  Published
    Property Account: String Read FAccount Write FAccount;
    Property CanHardStop: Boolean Read FCanHardStop Write FCanHardStop Default True;
    Property CanPause: Boolean Read FCanPause Write FCanPause Default True;
    Property DisplayName: String Read FDisplayName Write FDisplayName;
    Property Dependencies: String Read GetDependencies Write SetDependencies;
    Property Interactive: Boolean Read FInteractive Write FInteractive;
    Property Password: String Read FPassword Write FPassword;
    Property ServiceName: String Read FServiceName Write SetServiceName;
    Property ServiceType: TServiceType Read FServiceType Write FServiceType Default stWin32;
    Property StartType: DWord Read FStartType Write FStartType Default SERVICE_DEMAND_START;
    Property WaitHint: Integer Read FWaitHint Write FWaitHint Default 1000;
  End;

Procedure CmdLine;

Function AsPChar (Var Value: String): PChar;

Var
  ServiceList: TList;

Implementation

{- helper functions }

Function AsPChar (Var Value: String): PChar;
Begin
  If Value='' Then
    Result:= Nil
  Else
    Result:= PChar(Value)
End;

{- TWatchDog helper thread }

Type
  TWatchDog = Class(TThread)
  Public
    Service: TService;
    Constructor Create (AService: TService);
    Procedure Execute; Override;
  End;

Constructor TWatchDog.Create (AService: TService);
Begin
  Inherited Create(True);
  Service:= AService;
  FreeOnTerminate:= True;
  Resume
End;

Procedure TWatchDog.Execute;
Var
  i: Integer;
Begin
  i:= 0;
  While (i<5) And (WaitForOneObject([Service], Service.WaitHint)<>WAIT_OBJECT_0) Do Begin
    Service.Status:= SERVICE_STOP_PENDING;
    Inc(i)
  End;
  If Service.Status<>SERVICE_STOPPED Then Begin
    { the service did not respond, we have to kill the thread }
    If TerminateThread(Service.Handle, 0) Then Begin
      Service.AddToEventLog(elWarning, Format('%s had to be terminated: it did not respond to the stop request',
        [Service.ServiceName]));
      Service.Status:= SERVICE_STOPPED
    End
  End
End;

{- TService}

Constructor TService.Create (Const AServiceName, ADisplayName: String);
Begin
  Inherited Create(True);
  EventLog:= TEventLog.Create(AServiceName);
  StopEvent:= TEvent.Create;
  FMainWrapper:= CreateWrapper(Self, @TService.Main);
  FCtrlWrapper:= CreateWrapper(Self, @TService.Ctrl);
  ServiceName:= AServiceName;
  DisplayName:= ADisplayName;
  CanPause:= True;
  CanHardStop:= True; {allow WatchDog thread to kill service if it is not responding }
  ServiceType:= stWin32;
  StartType:= SERVICE_DEMAND_START;
  WaitHint:= 1000;
  ServiceList.Add(Self)
End;

Destructor TService.Destroy;
Begin
  StopEvent.Signal; {is it necessary or should Event.Free signal?}
  StopEvent.Free;
  Inherited Destroy
End;

Procedure TService.Check (Value: Bool);
Begin
  If Not Value Then
    Raise EService.Create('EService failure') At ReturnAddr
End;

Procedure TService.Start;
Begin
End;

Procedure TService.Execute;
Begin
  {-this waits much more efficiently than checking for 'Terminated'}
  WaitForOneObject([StopEvent], Infinite)
End;

Procedure TService.InternalStop;
Var
  i: Integer;
Begin
  Status:= SERVICE_STOP_PENDING;
  If Suspended Then
    Resume;
  Terminate;
  StopEvent.Signal;
  If (WaitForOneObject([Self], WaitHint)<>WAIT_OBJECT_0) And CanHardStop Then Begin
    Status:= SERVICE_STOP_PENDING;
    {spawn thread to watch the termination of our service}
    TWatchDog.Create(Self)
  End;
  {AfterExecute;}
End;

Procedure TService.InternalPause;
Begin
  Status:= SERVICE_PAUSE_PENDING;
  Self.Suspend;
  Status:= SERVICE_PAUSED;
End;

Procedure TService.InternalContinue;
Begin
  Status:= SERVICE_CONTINUE_PENDING;
  Resume;
  Status:= SERVICE_RUNNING
End;

Procedure TService.Stop;
Begin
End;

Function TService.GetDependencies: String;
Begin
  Result:= StrPas(FDependencies)
End;

Procedure TService.SetServiceName (Const Value: String);
Begin
  FServiceName:= Value;
  EventLog.Name:= Value
End;

Procedure TService.SetDependencies (Const Value: String);
Begin
  StrPLCopy(FDependencies, Value, SizeOf(FDependencies)-1)
End;

Function TService.GetStatus: Integer;
Begin
  Result:= FStatus.dwCurrentState
End;

Function TService.GetStatusHandle: TService_Status_Handle;
Begin
  If FStatusHandle=0 Then Begin
    FStatusHandle:= RegisterServiceCtrlHandler(PChar(FServiceName), @FCtrlWrapper);
    If FStatusHandle=0 Then
      Raise EService.Create('Unable to register service control handler')
  End;
  Result:= FStatusHandle
End;

Procedure TService.SetStatus (Value: Integer);
Begin
  If FStatusHandle=0 Then
    InitStatus;
  FStatus.dwCurrentState:= Value;
  Case Value Of
    SERVICE_STOP_PENDING: FStatus.dwWaitHint:= 20; {according to SDK doc}
    SERVICE_STOP,
    SERVICE_RUNNING: FStatus.dwWaitHint:= 0;
  Else
    FStatus.dwWaitHint:= WaitHint
  End;

  If (Value = SERVICE_START_PENDING) Then
    FStatus.dwControlsAccepted:= 0
  Else Begin
    FStatus.dwControlsAccepted:= SERVICE_ACCEPT_STOP;
    If CanPause Then
      FStatus.dwControlsAccepted:= FStatus.dwControlsAccepted
                                Or SERVICE_ACCEPT_PAUSE_CONTINUE
  End;
  ReportStatus
End;

Procedure TService.InitStatus;
{- preset FStatus}
Begin
  FillChar(FStatus, SizeOf(FStatus), 0);
  FStatus.dwServiceType:= ProcessType;
  FStatus.dwServiceSpecificExitCode:= 0;
  FStatus.dwWin32ExitCode:= NO_ERROR;
  FStatus.dwCheckPoint:= 0;
End;

Function TService.ProcessType: Integer;
Begin
  Case ServiceType Of
    stKernel:     Result:= SERVICE_KERNEL_DRIVER;
    stFileSystem: Result:= SERVICE_FILE_SYSTEM_DRIVER;
  Else
    If ServiceList.Count>1 Then
      Result:= SERVICE_WIN32_SHARE_PROCESS
    Else
      Result:= SERVICE_WIN32_OWN_PROCESS;
    If Interactive And (Account='') Then
      Result:= Result Or SERVICE_INTERACTIVE_PROCESS;
  End;
End;

{- report current Status to Service Control Manager }

Procedure TService.ReportStatus;
Begin
  If Debug Then
    Exit;

  Case FStatus.dwCurrentState Of
    SERVICE_START_PENDING,
    SERVICE_PAUSE_PENDING,
    SERVICE_STOP_PENDING: Inc(FStatus.dwCheckPoint);
  Else
    FStatus.dwCheckPoint:= 0 {has to be zero if none of the above}
  End;
  If Not SetServiceStatus(StatusHandle, FStatus) Then
    Raise EService.Create('SetServiceStatus');
  FStatus.dwWaitHint:= 0 {reset wait hint}
End;

Procedure TService.Ctrl (Code: Integer);
Begin
  Try
    Case Code Of
      SERVICE_CONTROL_STOP:     InternalStop;
      SERVICE_CONTROL_PAUSE:    InternalPause;
      SERVICE_CONTROL_CONTINUE: InternalContinue;
    Else
      ReportStatus
    End
  Except
    On E: Exception Do
      AddToEventLog(elException, Format('%s raised %s: %s', [ServiceName, E.ClassName, E.Message]))
  End
End;

Procedure TService.Main (NumArgs: DWord; Var Args: PChar);
{notice this function is not running in the TThread context!}
Begin
  Try
    Status:= SERVICE_START_PENDING; {calls Ctrl through GetStatusHandle too}
    StopEvent.Reset;
    Start;
    Resume;
    Status:= SERVICE_RUNNING;
    {because we are not running in the TThread context, we can wait
     for Self (the thread) to finish}
    WaitForOneObject([Self], Infinite);
    FStatus.dwWin32ExitCode:= GetLastError;
    Status:= SERVICE_STOPPED
  Except
    On E: Exception Do
      AddToEventLog(elException, Format('%s raised %s: %s', [ServiceName, E.ClassName, E.Message]))
  End
End;

Function TService.ServiceEntry: TService_Table_Entry;
Begin
  Result.lpServiceName:= PChar(FServiceName);
  Result.lpServiceProc:= @FMainWrapper
End;

Function TService.GetLastErrorText: String;
Var
  dwRet: Integer;
  szTemp: Array[0..511] Of Char;
Begin
  Result:= '';
  dwRet:= FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM Or FORMAT_MESSAGE_ARGUMENT_ARRAY,
                        Nil, GetLastError, LANG_NEUTRAL,
                        szTemp, SizeOf(szTemp)-2, Nil);

  szTemp[StrLen(szTemp)-2]:= #0;  //remove cr and newline character
  Result:= Format('%s (0x%x)', [szTemp, GetLastError])
End;

Procedure TService.AddToEventLog (AType: TEventLogType; Value: String);
Begin
  If Debug Then
    WriteLn(Value)
  Else
    EventLog.Add(AType, Value)
End;

{- installation stuff --------------------------------------------------}

Procedure TService.Install;
Var
  schService,
  schSCManager: TSC_Handle;
  ModuleName: Array[0..254] Of Char;
Begin
  If (GetModuleFileName(0, ModuleName, SizeOf(ModuleName))= 0) Then Begin
    WriteLn(Format('Unable to install %s - %s', [FDisplayName, GetLastErrorText]));
    Exit
  End;
  schSCManager:= OpenSCManager(Nil, Nil, SC_MANAGER_ALL_ACCESS);
  If schSCManager<>0 Then Begin
    schService:= CreateService(schSCManager, PChar(FServiceName), PChar(FDisplayName),
                               SERVICE_ALL_ACCESS, ProcessType,
                               FStartType, SERVICE_ERROR_NORMAL,
                               ModuleName, Nil, Nil, FDependencies,
                               AsPChar(FAccount), AsPChar(FPassword));
     if schService<>0 Then Begin
       WriteLn(Format('%s installed.', [FDisplayName]));
       CloseServiceHandle(schService);
       If Not EventLog.Register(ModuleName) Then
         WriteLn('Warning: could not register with event logging')
     End Else
       WriteLn(Format('CreateService failed - %s', [GetLastErrorText]));
     CloseServiceHandle(schSCManager)
  End Else
    WriteLn(Format('OpenSCManager failed - %s', [GetLastErrorText]))
End;

Procedure TService.Remove;
Var
  schService,
  schSCManager: TSC_Handle;
Begin
  schSCManager:= OpenSCManager(Nil, Nil, SC_MANAGER_ALL_ACCESS);
  If schSCManager<>0 Then Begin
    schService:= OpenService(schSCManager, PChar(FServiceName), SERVICE_ALL_ACCESS);
    If schService<>0 Then Begin
      { try to stop the service }
      If ControlService(schService, SERVICE_CONTROL_STOP, FStatus) Then Begin
        Write(Format('Stopping %s', [FDisplayName]));
        // Sleep(1000);
        While QueryServiceStatus(schService, FStatus) Do Begin
          If FStatus.dwCurrentState = SERVICE_STOP_PENDING Then Begin
            Write('.');
            Sleep(500)
          End Else
            Break
        End;
        If FStatus.dwCurrentState = SERVICE_STOPPED  Then
          WriteLn(Format(#13#10'%s stopped.', [FDisplayName]))
        Else
          WriteLn(Format(#13#10'%s failed to stop.', [FDisplayName]))
      End;

      { now remove the service }
      If DeleteService(schService) Then Begin
        WriteLn(Format('%s removed.', [FDisplayName]));
        If Not EventLog.UnRegister Then
          WriteLn('Warning: could not delete registry key for event logging')
      End Else
        WriteLn(Format('DeleteService failed - %s.', [GetLastErrorText]));
      CloseServiceHandle(schService)
    End Else
      WriteLn(Format('OpenService failed - %s.', [GetLastErrorText]));
    CloseServiceHandle(schSCManager)
  End Else
    WriteLn(Format('OpenSCManager failed - %s.', [GetLastErrorText]))
End;

Function DebugConsoleHandler (dwCtrlType: Integer): Bool; StdCall;
Var
  i: Integer;
Begin
  Result:= False;
  Case dwCtrlType Of
    CTRL_BREAK_EVENT,
    CTRL_C_EVENT: Begin
      For i:= 0 To ServiceList.Count-1 Do With TService(ServiceList[i]) Do Begin
        WriteLn('Stopping ', DisplayName);
        InternalStop
      End;
      Result:= True
    End
  End
End;

Procedure Debug;
Var
  i: Integer;
Begin
  SetConsoleCtrlHandler(@DebugConsoleHandler, True);
  For i:= 0 To ServiceList.Count-1 Do With TService(ServiceList[i]) Do Begin
    Debug:= True;
    WriteLn('Starting ', DisplayName);
    Start;
    Resume
  End;
  WriteLn('Press <Ctrl>-Break to terminate');
  {-wait for all our threads to end}
  WaitForAllInList(ServiceList, Infinite)
End;

Procedure CmdLine;
Type
  PServiceTable = ^TServiceTable;
  TServiceTable = Array[0..0] Of TService_Table_Entry;
Var
  Cmd: ShortString;
  dispatchTable: PServiceTable;
  i, NoS: Integer;
Begin
  NoS:= ServiceList.Count;
  If NoS<1 Then
    Raise Exception.Create('No services to handle.');

  Cmd:= UpperCase(ParamStr(1));
  If (Length(cmd)>0) And (cmd[1]='/') Then
    cmd[1]:= '-';

  If Cmd='-INSTALL' Then
    For i:= 0 To NoS-1 Do
      TService(ServiceList[i]).Install
  Else
  If Cmd='-REMOVE' Then
    For i:= 0 To NoS-1 Do
      TService(ServiceList[i]).Remove
  Else
  If Cmd='-DEBUG' Then
    Debug
  Else Begin
    GetMem(dispatchTable, (NoS+1)*SizeOf(TService_Table_Entry));
    Try
      For i:= 0 To NoS-1 Do
        dispatchTable^[i]:= TService(ServiceList[i]).ServiceEntry;
      dispatchTable[NoS].lpServiceName:= Nil;
      dispatchTable[NoS].lpServiceProc:= Nil;
      If Not StartServiceCtrlDispatcher(dispatchTable^) Then
        Raise Exception.Create('StartServiceCtrlDispatcher failed.');
    Finally
      FreeMem(dispatchTable, (NoS+1)*SizeOf(TService_Table_Entry))
    End
  End;
  Halt(0)
End;

Procedure FreeServiceList;
Var
  i: Integer;
Begin
  For i:= 0 To ServiceList.Count-1 Do
    TService(ServiceList[i]).Free;
  ServiceList.Free
End;

Initialization
  ServiceList:= TList.Create;
Finalization
  FreeServiceList;
End.
