unit newreg;
{$R-}
interface
uses Windows, Classes, SysUtils, Registry, secureobj;

Type

   TSecureReg = class(TSecureableObject)
   private
    FKeyHandle: HKEY;
   protected
    procedure SetObjectSecurity(hObject: HKEY);
    procedure GetObjectSecurity(hObject: HKEY);
    procedure SetTheDescriptor;override;//these were the abstract methods from TSecObject
    procedure GetTheDescriptor;override;
   public
    property KeyHandle: HKEY Read FKeyHandle Write FKeyHandle;
  end;

  TNewReg = class(TRegistry)
  private
    FSecure: TSecureReg;
  protected
    FFileName: string;
  public
    constructor Create(const FileName: string; const rKey: HKEY);virtual;
    Destructor Destroy;override;
    function ReadString(const Section, Ident, Default: string): string;
    procedure WriteString(const Section, Ident, Value: String);
    function ReadInteger(const Section, Ident: string;
      Default: Longint): Longint;
    procedure WriteInteger(const Section, Ident: string; Value: Longint);
    function ReadBool(const Section, Ident: string; Default: Boolean): Boolean;
    procedure WriteBool(const Section, Ident: string; Value: Boolean);
    procedure ReadSection(const Section: string; Strings: TStrings);
    procedure ReadSections(Strings: TStrings);
    procedure ReadSectionValues(const Section: string; Strings: TStrings);
    procedure EraseSection(const Section: string);
    procedure DeleteKey(const Section, Ident: String);
    property FileName: string read FFileName;
    property Security: TSecureReg Read FSecure Write FSecure;
  end;


implementation
uses Consts;

// helper functions
procedure ReadError(const Name: string);
begin
  raise ERegistryException.CreateResFmt(SInvalidRegType, [Name]);
end;

function IsRelative(const Value: string): Boolean;
begin
  Result := not ((Value <> '') and (Value[1] = '\'));
end;

function RegDataToDataType(Value: TRegDataType): Integer;
begin
  case Value of
    rdString: Result := REG_SZ;
    rdExpandString: Result := REG_EXPAND_SZ;
    rdInteger: Result := REG_DWORD;
    rdBinary: Result := REG_BINARY;
  else
    Result := REG_NONE;
  end;
end;

function DataTypeToRegData(Value: Integer): TRegDataType;
begin
  if Value = REG_SZ then Result := rdString
  else if Value = REG_EXPAND_SZ then Result := rdExpandString
  else if Value = REG_DWORD then Result := rdInteger
  else if Value = REG_BINARY then Result := rdBinary
  else Result := rdUnknown;
end;

{TSecureReg}

procedure TSecureReg.SetTheDescriptor;
begin
 if KeyHandle = 0 then
    raise ESecObjError.Create('Invalid Key Handle', ERROR_INVALID_HANDLE);
 SetObjectSecurity(KeyHandle);
end;

procedure TSecureReg.GetTheDescriptor;
begin
 if KeyHandle = 0 then
    raise ESecObjError.Create('Invalid Key Handle', ERROR_INVALID_HANDLE);
 GetObjectSecurity(KeyHandle);
end;

procedure TSecureReg.SetObjectSecurity(hObject: HKEY);
var
 si: SECURITY_INFORMATION;
 ret: LongInt;
begin
si := DACL_SECURITY_INFORMATION;// Or SACL_SECURITY_INFORMATION;
TRY
 ret := RegSetKeySecurity(hObject,si,m_pSD);
 if ( ret <> ERROR_SUCCESS) then
    raise ESecObjError.Create('RegSetKeySecurity Failed', ret);
FINALLY
 FreeDataStructures;
END;
end;

procedure TSecureReg.GetObjectSecurity(hObject: HKEY);
var
 pSelfRelativeReturnSD: PSecurityDescriptor;
 iSDSize: DWORD;
 si: SECURITY_INFORMATION;
 ret: LongInt;
begin
 pSelfRelativeReturnSD := nil; iSDSize := 0; si := DACL_SECURITY_INFORMATION;

 ret := RegGetKeySecurity(hObject, si,pSelfRelativeReturnSD,iSDSize);
 if (ret <> ERROR_SUCCESS) And ( ret <> ERROR_INSUFFICIENT_BUFFER) then
    raise ESecObjError.Create('RegGetKeySecurity Failed', ret);
 if ret = ERROR_INSUFFICIENT_BUFFER then begin
   GetMem(pSelfRelativeReturnSD, iSDSize);
 TRY
  if (pSelfRelativeReturnSD = nil) then
     raise ESecObjError.Create('RegGetKeySecurity Failed', ERROR_NOT_ENOUGH_MEMORY);
  if (RegGetKeySecurity(hObject, si,pSelfRelativeReturnSD,iSDSize) <> ERROR_SUCCESS) Then
     raise ESecObjError.Create('RegGetKeySecurity Failed', GetLastError);
  BuildSD(pSelfRelativeReturnSD);
 FINALLY
  if pSelfRelativeReturnSD <> nil then FreeMem(pSelfRelativeReturnSD);
 END;
 end
 else
  BuildSD(pSelfRelativeReturnSD);
end;

{TNewReg}

constructor TNewReg.Create(const FileName: string; const rKey: HKEY);
begin
  inherited Create;
  RootKey := rKey;
  FFileName := FileName;
  OpenKey(FileName, True);
  FSecure := TSecureReg.Create(False);
  FSecure.KeyHandle := CurrentKey;
end;

destructor TNewReg.Destroy;
begin
 FSecure.Free;
 inherited Destroy;
end;

function TNewReg.ReadString(const Section, Ident, Default: string): string;
var
  Key, OldKey: HKEY;
begin
  Key := GetKey(Section);
  if Key <> 0 then
  try
    OldKey := CurrentKey;
    SetCurrentKey(Key);
    try
      if ValueExists(Ident) then
        Result := inherited ReadString(Ident) else
        Result := Default;
    finally
      SetCurrentKey(OldKey);
    end;
  finally
    RegCloseKey(Key);
  end
  else Result := Default;
end;

procedure TNewReg.WriteString(const Section, Ident, Value: String);
var
  Key, OldKey: HKEY;
begin
  CreateKey(Section);
  Key := GetKey(Section);
  if Key <> 0 then
  try
    OldKey := CurrentKey;
    SetCurrentKey(Key);
    try
      inherited WriteString(Ident, Value);
    finally
      SetCurrentKey(OldKey);
    end;
  finally
    RegCloseKey(Key);
  end;
end;

function TNewReg.ReadInteger(const Section, Ident: string; Default: LongInt): LongInt;
var
  Key, OldKey: HKEY;
  S: string;
begin
  Key := GetKey(Section);
  if Key <> 0 then
  try
    OldKey := CurrentKey;
    SetCurrentKey(Key);
    try
      if ValueExists(Ident) then
      begin
        S := inherited ReadString(Ident);
        if (Length(S) > 2) and (S[1] = '0') and (UpCase(S[2]) = 'X') then
          S := '$' + Copy(S, 3, Maxint);
        Result := StrToIntDef(S, Default);
      end else
        Result := Default;
    finally
      SetCurrentKey(OldKey);
    end;
  finally
    RegCloseKey(Key);
  end
  else Result := Default;
end;

procedure TNewReg.WriteInteger(const Section, Ident: string; Value: LongInt);
var
  Key, OldKey: HKEY;
begin
  CreateKey(Section);
  Key := GetKey(Section);
  if Key <> 0 then
  try
    OldKey := CurrentKey;
    SetCurrentKey(Key);
    try
      inherited WriteString(Ident, IntToStr(Value));
    finally
      SetCurrentKey(OldKey);
    end;
  finally
    RegCloseKey(Key);
  end;
end;

function TNewReg.ReadBool(const Section, Ident: string; Default: Boolean): Boolean;
begin
  Result := ReadInteger(Section, Ident, Ord(Default)) <> 0;
end;

procedure TNewReg.WriteBool(const Section, Ident: string; Value: Boolean);
const
  Values: array[Boolean] of string = ('0', '1');
var
  Key, OldKey: HKEY;
begin
  CreateKey(Section);
  Key := GetKey(Section);
  if Key <> 0 then
  try
    OldKey := CurrentKey;
    SetCurrentKey(Key);
    try
      inherited WriteString(Ident, Values[Value]);
    finally
      SetCurrentKey(OldKey);
    end;
  finally
    RegCloseKey(Key);
  end;
end;

procedure TNewReg.ReadSection(const Section: string; Strings: TStrings);
var
  Key, OldKey: HKEY;
begin
  Key := GetKey(Section);
  if Key <> 0 then
  try
    OldKey := CurrentKey;
    SetCurrentKey(Key);
    try
      inherited GetValueNames(Strings);
    finally
      SetCurrentKey(OldKey);
    end;
  finally
    RegCloseKey(Key);
  end;
end;

procedure TNewReg.ReadSections(Strings: TStrings);
begin
  GetKeyNames(Strings);
end;

procedure TNewReg.ReadSectionValues(const Section: string; Strings: TStrings);
var
  KeyList: TStringList;
  I: Integer;
begin
  KeyList := TStringList.Create;
  try
    ReadSection(Section, KeyList);
    Strings.BeginUpdate;
    try
      for I := 0 to KeyList.Count - 1 do
        Strings.Values[KeyList[I]] := ReadString(Section, KeyList[I], '');
    finally
      Strings.EndUpdate;
    end;
  finally
    KeyList.Free;
  end;
end;

procedure TNewReg.EraseSection(const Section: string);
begin
  inherited DeleteKey(Section);
end;

procedure TNewReg.DeleteKey(const Section, Ident: String);
begin
  inherited DeleteValue(Ident);
end;

end.
