/*
	File:		DILCPipe.h

	Contains:	C Definitions of the DIL Pipe class

	Written by:	Rob Langhorne

	Copyright:	c 1995 by Apple Computer, Inc.  All rights reserved.

	Change History (most recent first): 
			version 1.0 Universal CDIL Headers
	To Do:

*/

#ifndef __DILCPIPE_H
#define __DILCPIPE_H

#if GENERATINGPOWERPC
#pragma options align=mac68k
#endif

/*-------------------------------------------------------------------------
**
**	Definitions
**
** ------------------------------------------------------------------------ */

/*-------------------------------------------------------------------------
**	Compatibility definitions	
** ------------------------------------------------------------------------ */
#ifdef forMac
	#define FAR
	#define PASCAL
	#define _export 
	
	#include <Types.h>
#else
	typedef unsigned char Boolean;
	#define nil NULL
	#define true TRUE
	#define false FALSE
	typedef long Size;

	#ifndef forWin16
		#define _export 
	#endif

#endif

/*-------------------------------------------------------------------------
**	Local Error Codes	
** ------------------------------------------------------------------------ */

#define kDILErrBase 			(-700 + -28000)			/*	Error code base number				*/

#define kCommErrNoErr			0						/*	No error							*/
#define kOutOfMemory			(-1 + kDILErrBase)		/*	Error on memory allocation			*/
#define kBadPipeState			(-2 + kDILErrBase)		/*	DIL Pipe was set to a bad state		*/
#define kExceptionErr			(-3 + kDILErrBase)		/*	An unhandled exception was raised	*/
#define kQueueFullError			(-4 + kDILErrBase)		/*	async completion queue is full		*/
#define kPipeNotInitialized		(-5 + kDILErrBase)		/*	Pipe has not been initialized		*/
#define kInvalidParameter		(-6 + kDILErrBase)		/*	A parameter passed in was invalid	*/
#define kPipeNotReady			(-7 + kDILErrBase)		/*	Pipe is not ready for operation		*/
#define kTimeout				(-8 + kDILErrBase)		/* 	A timeout has occurred				*/

/*-------------------------------------------------------------------------
**	CDIL State type	
**		The following enumeration is used to identify the state of the
**		current CDIL Pipe.
** ------------------------------------------------------------------------ */

typedef enum CDIL_State {
	kCDIL_Uninitialized = 0,	/*	0CDIL is uninitialized												*/
	kCDIL_InvalidConnection,	/*  1CDIL tried to bring up connection, but it failed					*/
	kCDIL_Disconnected,			/*	2CDIL is not connected												*/
	kCDIL_Listening,			/*	3CDIL is listening for a connection									*/
	kCDIL_ConnectPending,		/*	4A connection is pending												*/
	kCDIL_Connected,			/*	5CDIL is connected													*/
	kCDIL_Busy,					/*	6CDIL is either reading or writing								 	*/
	kCDIL_Aborting,				/*	7CDIL is currently aborting											*/
	kCDIL_Startup,				/*	8CDIL is currently starting up										*/
	kCDIL_Userstate = 50		/*	50Users can add CDIL states after this point							*/
} CDIL_State;

/*-------------------------------------------------------------------------
**	Unicode encoding definitions (from Unicode.h) -> Just documenting the 
**				currently available values here
typedef enum {
	kUnicode = 0,				// not implemented
	kMacRomanEncoding,			// Macintosh Roman
	kPCRomanEncoding = 3,		// PC Roman
} TCharEncoding;
** ------------------------------------------------------------------------ */

/*-------------------------------------------------------------------------
**	Pipe direction definitions	
** ------------------------------------------------------------------------ */

typedef enum CDILPipeDirection {
	kUndefinedDirection = 0,
	kReadPipe,					/*	Read pipe direction						*/
	kWritePipe,					/*	Write pipe direction					*/
	kAllPipes					/*	Apply to all pipes (where applicable)	*/
} CDILPipeDirection;

typedef void CDILPipe;

/*-------------------------------------------------------------------------
**	Other definitions	
** ------------------------------------------------------------------------ */
#ifdef forMac
	#define kDefaultTimeout 	60*30	/*	Default timeout for DIL operations	*/
#else	// for Windows
	#define kDefaultTimeout 	1000	/*	Default timeout for DIL operations	*/
#endif
#define	kDefaultBufferSize	1024		/*	Default buffer size for DIL I/O		*/

typedef long CommErr;					/* Comm Error Code						*/

/*------------------------------------------------------------------------------------
**	Completion function definitions
**		Completion functions and other function pointers that are used by CDILs
**		should conform to one of the following type definitions.  
**		
**		These are:
**			CDILCompletionProcPtr -  the basic callback function definition.  This
**									is used by CDPipeListen() and CDPipeWrite().
**		
**			CDILPipeCompletionProcPtr - the read callback function definition.  Any
**									data returned is in the buffer pointed to by
**									the pData parameter.  This is used by CDPipeRead().
**		
**			CDILEncryptionProcPtr -  This points to an encryption callback hook.  
**									This can be used to encrypt the data stream using
**									the user-defined algorithm just prior to 
**									putting the data on the wire.
**		
**			CDILDecryptionProcPtr -  This points to an decryption callback hook.  
**									This can be used to decrypt the data stream using
**									the user-defined algorithm just after the 
**									data is pulled off the wire.
** ----------------------------------------------------------------------------------- */
typedef void (FAR *CDILCompletionProcPtr)( CommErr errorValue, long refCon ) ;
typedef void (FAR *CDILPipeCompletionProcPtr)( CommErr errorValue, void FAR *pData, short Count, long refCon, long lFlags ) ;
typedef CommErr (FAR *CDILEncryptionProcPtr)( void FAR *pData, short Count, long refCon ) ;
typedef CommErr (FAR *CDILDecryptionProcPtr)( void FAR *pData, short Count, long refCon ) ;

/*
**	These definitions will result in functions that look like the following:
**	
**	void CompletionHook 		( CommErr errorValue, long refCon ) ;
**	void PipeCompletionHook 	( CommErr errorValue, void *pData, Size Count, long refCon, long lFlags ) ;
**	void EncryptHook 			( void *pData, Size Count, long refCon) ;
**	void DecryptHook 			( void *pData, Size Count, long refCon) ;
*/

		/* ~~~~~~~ CDIL initialization and cleanup methods		*/

#ifdef __cplusplus
extern "C" {
		CommErr 	FAR PASCAL CDPipeInit	( CDILPipe FAR *pipe, const char FAR *connectionType, const char FAR *connectionName, 
											  char FAR * configInfo, char FAR * pipePort, Size  readSize = kDefaultBufferSize, 
											  Size  writeSize = kDefaultBufferSize ) ;
#else
		CommErr 	FAR PASCAL CDPipeInit	( CDILPipe FAR *pipe, char FAR *connectionType, char FAR *connectionName, char FAR * configInfo,
											  char FAR * pipePort, Size  readSize, Size  writeSize ) ;
#endif

		CommErr 	FAR  PASCAL CDPipeDisconnect ( CDILPipe FAR *pipe  ) ;
#ifdef __cplusplus
		CommErr		FAR  PASCAL CDPipeListen 	( CDILPipe FAR *pipe, long timeout = kDefaultTimeout, CDILCompletionProcPtr completionHook = nil, long refCon = 0 ) ;
#else
		CommErr		FAR  PASCAL CDPipeListen 	( CDILPipe FAR *pipe, long timeout, CDILCompletionProcPtr completionHook, long refCon ) ;
#endif

		CommErr		FAR  PASCAL CDPipeAccept 	( CDILPipe FAR *pipe  ) ;
#ifdef __cplusplus
		CommErr		FAR  PASCAL CDPipeAbort 	( CDILPipe FAR *pipe, CDILPipeDirection direction = kAllPipes ) ;
#else
		CommErr		FAR  PASCAL CDPipeAbort 	( CDILPipe FAR *pipe, CDILPipeDirection direction ) ;
#endif

		/* ~~~~~~~ CDIL read and write methods					*/

#ifdef __cplusplus
		CommErr 	FAR PASCAL  CDPipeRead	( CDILPipe FAR *pipe, void FAR * p, long FAR * count, Boolean FAR * eom, long swapSize = 0, 
											long destEncoding = 0, long timeOut = 0, CDILPipeCompletionProcPtr completionHook = nil, 
											long refCon = 0) ;
		CommErr 	FAR  PASCAL CDBytesInPipe( CDILPipe FAR *pipe, CDILPipeDirection direction, long FAR *count ) ;
		CommErr 	FAR  PASCAL CDPipeWrite	( CDILPipe FAR *pipe, void FAR * p, long FAR * count, Boolean eom = true, long swapSize = 0, long srcEncoding = 0, 
			   						  		long timeOut = 0, CDILCompletionProcPtr completionHook = nil, long refCon = 0) ;
#else
		CommErr 	FAR  PASCAL CDPipeRead	( CDILPipe FAR *pipe, void FAR * p, long FAR * count, Boolean FAR * eom, long swapSize, long destEncoding, 
			   						  		long timeOut, CDILPipeCompletionProcPtr completionHook, long refCon) ;
		CommErr 	FAR  PASCAL CDBytesInPipe	( CDILPipe FAR *pipe, CDILPipeDirection direction, long FAR *count ) ;
		CommErr 	FAR  PASCAL CDPipeWrite	( CDILPipe FAR *pipe, void FAR * p, long FAR* count, Boolean eom, long swapSize, long srcEncoding, 
			   						 		long timeOut, CDILCompletionProcPtr completionHook, long refCon) ;
#endif


		/* ~~~~~~~ CDIL maintenance methods						*/

		void		FAR  PASCAL CDIdle 			( CDILPipe FAR *pipe ) ;
		CDIL_State	FAR  PASCAL CDGetPipeState 	( CDILPipe FAR *pipe  ) ;
		CommErr		FAR  PASCAL CDSetPipeState	( CDILPipe FAR *pipe, CDIL_State user_state ) ;

		/* ~~~~~~~ CDIL encryption definition methods				*/

		void		FAR  PASCAL CDEncryptFunction	( CDILPipe FAR *pipe, CDILEncryptionProcPtr encryptFunction, long refCon ) ;
		void		FAR  PASCAL CDDecryptFunction	( CDILPipe FAR *pipe, CDILDecryptionProcPtr decryptFunction, long refCon ) ;

		/* ~~~~~~~ CDIL Information methods						*/
		char FAR * 	FAR  PASCAL CDGetConfigStr 	( CDILPipe FAR *pipe ) ;
		char FAR *	FAR  PASCAL CDGetPortStr	( CDILPipe FAR *pipe ) ;
		long 	 	FAR  PASCAL CDGetTimeout 	( CDILPipe FAR *pipe ) ;
#ifndef forMac
		void 		FAR  PASCAL CDSetApplication   ( CDILPipe FAR *pipe, short appInstance );
#endif
		/* ~~~~~~~ CDIL advanced methods							*/
		
		CommErr 	FAR  PASCAL CDFlush 					( CDILPipe FAR *pipe, CDILPipeDirection direction ) ;
		void 			FAR  PASCAL CDPad 						( CDILPipe FAR *pipe, long length ) ;
		void			FAR PASCAL CDSetPadState		( CDILPipe FAR *pipe, Boolean  paddingOn ) ;
#ifdef __cplusplus
	}	//	extern "C"
#endif

/*-------------------------------------------------------------------------
**		Basic functions
**			These functions are used at application startup or shutdown to 
**			perform global preparation/cleanup of the DIL environment.
** ------------------------------------------------------------------------ */

#ifdef __cplusplus
extern "C" {
#endif
		CommErr 	FAR PASCAL CDInitCDIL 						( void ) ; 
		CommErr 	FAR PASCAL CDDisposeCDIL 				( void ) ;
		CDILPipe	FAR * PASCAL CDCreateCDILObject	( void ) ;
		CommErr		FAR PASCAL CDDisposeCDILObject 	(  CDILPipe FAR *pipe ) ;

#ifdef __cplusplus
}
#endif

#endif	/* __CDILPipe_H */
