 
PROTOCOL.C 1~            @@/          1~1~   @@TEXTMACA          )  ȭ:lY                         /* Protocol.c
*
*  Written by: Bill Worzel
*
*  Copyright:   1995-96 by Apple Computer, Inc.  All rights reserved.
*
* This file is used both by the Macintosh and Windows versions of DILArchive; it contains
* nearly all the calls to the CDIL code other than the initialization routines. It is based on
* the SoupDrink example written by Rob Langhorne and David Fedor of Apple Computer.
*
* *
* Purpose:
*    To demonstrate the basic parts of CDILs based on the DILArchive transport protocol.
* The Archive function is the main call for getting data being archived or dearchived
* from the Newton PDA or sending previously archived data from the desktop machine to the Newton PDA.
* Note that the added "End " in the receiving protocol is due to the fact that the Newton PDA prefers
* to disconnect first.
*
* The protocol for DILArchive is as follows (direction is from Newton PDA):
*
*                      Sending:
*                          <connect>
*                          Newton->desktop "Sending"
*                          Newton->desktop <data format>
*                          desktop->Newton "READY"
*                          Newton->desktop <length> (of string)
*                          Newton->desktop <data>
*                          desktop->Newton "CONFIRMED" or "ABORT"
*                          Newton->desktop "End " or "More" or "Abrt"
*                          <disconnect>
*
*                      Receiving:
*                          <connect>
*                          Newton->desktop "Receive"
*                          desktop->Newton "READY FOR <data format>"
*                          Newton->desktop "Ready"
*                          desktop->Newton <null-terminated string>
*                          Newton->desktop "Conf" or "Abrt"
*                          desktop->Newton "END ARCHIVE" or "MORE" or "ABORT"
*                          Newton->desktop "End "
*                          <disconnect>
*
*
*/

#ifndef _windows
#define forMac      // Codewarrior doesn't have a handy way to do this.
#endif


#include <string.h>
#include <stdio.h>


#ifdef forMac   // ****** stuff for MAC-OS applications ******
#include <Types.h>
#include <memory.h>
#include <Packages.h>
#include <Errors.h>
#include <quickdraw.h>
#include <fonts.h>
#include <dialogs.h>
#include <windows.h>
#include <menus.h>
#include <events.h>
#include <OSEvents.h>
#include <Desk.h>
#include <diskinit.h>
#include <OSUtils.h>
#include <resources.h>
#include <toolutils.h>
#include <AppleEvents.h>
#include <string.h>
#include <Strings.h>
#include <stdio.h>
#include <Types.h>
#include <memory.h>

#include "DILCPIPE.H"
#include "HLFDIL.H"
#include "DILArchive.h"

#define Timeout 60*10      // this is currently in ticks but should be in milliseconds
//

#else           // ****** stuff for Windows applications *********

#include <windows.h>
#include "DILCPIPE.H"
#include "HLFDIL.H"
#include "DILARCHI.H"
#define Timeout 10000    // time out is 1 second
typedef unsigned char   Str255[256];

#define CurrentTimeInSeconds() ((long)(GetTickCount() / 1000))

#endif

#include "Protocol.h"

extern CDILPipe     *ourPipe;
extern short        gXferMode;          // A hook in case we want to archive frames later - tracks what we are working with
extern heap_ptr		gMemArray[kMaxNumMemItems]; // array of memory items used for frame bindings
extern heap_ptr		*gCurMemItem;		// position of next item allocated off heap

FILE                *gArchFile=0;       // UNIX file pointer to file being archived to/from
DILObj				*gDILObjArray[kMaxNumDILObjs] = {0}; // list of HLFDIL objects used for dearchiving
DILObj				**gCurDILObj;		// position of next DILObj created

/* Archive
 *
 * Purpose: Cover function for archiving or dearchiving based on first message received from Newton.
 *          Calls DoArchive to write strings from Newton to disk or DoDearchive to read them and send to Newton.
 */
CommErr Archive()
{
    char  response[kMAXSTR];
    long        length;
    CommErr     fErr;
    short       num_tries=0;

    // put up a status dialog to report on the pipe state and the archive process
    CHECKDILERROR ( InitializePipe() ); // CHECKDILERROR is a macro defined in Protocol.h

    CreateStatusDlog ();

    // make the connection!
    CHECKDILERROR ( DoConnectPipe () );

    length = 7;
    CHECKDILERROR ( ReadFromPipe ( response, &length, 0) );

    response[length] = '\0'; // make it a C string
    if (strncmp( "Sending", response, 7 ) == 0)
        fErr = DoArchive ( response );
    else if (strcmp("Receive",response) == 0)
        fErr = DoDearchive ( response );
    else {
        CHECKDILERROR ( EndArchive ( kDeskAbortError, true ) ); // CHECKDILERROR makes sure we're all tidy before abandoning ship
    }
    return ( fErr );
}


/* DoArchive
 *
 * Purpose: Creates file and writes string(s) from Newton PDA to file.
 */
CommErr DoArchive ( char *response )
{
    long        length,str_len;
    CommErr     fErr=0;
    short       retries=0;
    DILObj		*thisObject = nil;

    length = 4;
    CHECKDILERROR ( ReadFromPipe ( response, &length, 0 ) );
    response[length] = '\0'; // make it C string
    if (strncmp("STR ", response, 4) == 0)
        gXferMode = kStrings;
    else // if not string then should be frame
        gXferMode = kFrames;
        // put up standard file dialog and create archive file
    CHECKDILERROR ( CreateArchiveFile( gXferMode ) );
    SetArchiveStatus ( kConnected );

    if (gXferMode == kFrames)  /*	create FDIL object outside of loop	*/
			;// *** CREATE FDIL OBJECT HERE
		
    /*  Initiate conversation; tell the Newton PDA that we're ready */
    length = 6; // send terminator too
    CHECKDILERROR( WriteToPipe( "READY\4", &length, 0));

    do { // Loop getting Newton PDA data
        SetArchiveStatus ( kArchiving );
        if ( gXferMode == kStrings ) {
            length = sizeof ( long );
#ifndef _windows
            fErr = ReadFromPipe ( (char *)&str_len, &length, 0 ); // ReadFromPipe is our routine to handle events while waiting for incoming
#else
            fErr = ReadFromPipe ( (char *)&str_len, &length, length ); // ReadFromPipe is our routine to handle events while waiting for incoming
#endif
            if ( !fErr )
                fErr = ReadFromPipe ( response, &str_len, 0 ); // first read got string length, now get string itself
                
            if ( !fErr ) {
	            response[str_len] = '\0'; // make it a C string for convenience
	            fErr = ArchiveString ( response );
	        }
        } else { // doing frames
			// *** ADD CALL TO GET FRAME HERE
			if ( !fErr )
				fErr = (CommErr)WriteFrameToFile ( thisObject );
  		}
  		
        if ( !fErr ) // send CONFIRMED if no error
        {
            SetArchiveStatus ( kConfirming );
            length = 10; // send null terminator too
            WriteToPipe ("CONFIRMED\4", &length, 0);
        } else
            break; // else abandon loop

        // look for More or End Archive message
        length = 4;
        fErr = ReadFromPipe ( response, &length, 0 );
        response[length] = '\0';
        if ( fErr )
            break;
    } while ( strcmp("More",response) == 0 );

    if ( !fErr )
    { // check what message we got - if we don't understand it, abort
        if ( strcmp ( "Abrt", response ) == 0 )
        {
            SetArchiveStatus ( kNewtAbortError );
            fErr = EndArchive ( fErr, true );
        } else if ( strcmp ( "End ", response ) == 0 ){
            SetArchiveStatus ( kEndArchive );
            fErr = EndArchive ( 0, true );
        } else { // protocol messed up - unknown message
            SetArchiveStatus ( kDeskAbortError );
            fErr = EndArchive ( fErr, true );
        }
    } else { // fErr from archiving
        SetArchiveStatus ( kDeskAbortError );
        EndArchive ( kDeskAbortError, true ); // unable to write string to disk
    }

	if ( thisObject )
		;// *** DESTROY FDIL OBJECT HERE
    CDPipeDisconnect(ourPipe);
    return fErr;
}


/* DoDearchive
 *
 * Purpose: Finds previously archived file, reads string(s) from it and sends to Newton PDA.
 */
CommErr DoDearchive ( char *response ){
    long        length;
    CommErr     fErr=0;
    char        msg_string[256], content_str[256];
    Boolean     first_time=true;

    CHECKDILERROR ( OpenArchiveFile() ); // Get file from user and open it for reading

    SetArchiveStatus ( kConnected );
    strcpy(msg_string, "READY FOR ");
    if ( gXferMode == kStrings )
        strcat ( msg_string, "STR " );
    else
        strcat ( msg_string, "FRAM" );
    length = strlen ( msg_string ) + 1;
    msg_string[length-1] = '\4'; // add terminator character then send what we're dearchiving (strings or frames)
    CHECKDILERROR (WriteToPipe(msg_string, &length, 0));
    length = 5;
    CHECKDILERROR ( ReadFromPipe ( response, &length, 0 ) );
    response[length] = '\0';
    if ( strcmp("Ready",response) == 0 )
    {
        if ( gXferMode == kStrings ) { // dearchiving strings
            while ( DearchiveString ( content_str ) != EOF ) { // Loop sending strings to Newton PDA
              SetArchiveStatus ( kDearchiving );
     		   if ( !first_time ) {
                    strcpy( msg_string, "MORE\4" );
                    length = 5L; // send terminator too
                    fErr = WriteToPipe(msg_string, &length, 0);
                    if ( fErr ) // Houston, we have a problem
             			break;
                } else
                    first_time = false;
                length = strlen ( content_str ) + 1;
                content_str[length-1] = '\4';
                fErr = WriteToPipe(content_str, &length, 0);
                if ( fErr )
                    break; // didn't go thru, abort
                SetArchiveStatus ( kConfirming );
                length = 4;
                ReadFromPipe ( response, &length, 0 );
                response[length] = '\0'; // make it a C string for convenience
                if ( strcmp ( "Abrt", response ) == 0 ) {
                    fErr = kNewtAbortError;
                    break;
                } else if ( strcmp ( "Conf", response ) != 0 ) { // if not "Conf" protocol is messed up!
                    fErr = kDeskAbortError;
                    break;
                }
            } // end DearchiveString while
        } else {
            // DearchiveFrame creates needed DIL objects and binds them to nextFrame
        	DILObj	*nextFrame;
        	
        	gCurMemItem = gMemArray;
        	gCurDILObj = gDILObjArray;
			// *** CREATE FDIL OBJECT
       		
       		while ( !(fErr=DearchiveFrame ( nextFrame, false )) ) // will generate a warning
       		{
       			if ( !first_time ) {
                    strcpy( msg_string, "MORE\4" );
                    length = 5L; // send terminator too
                    fErr = WriteToPipe(msg_string, &length, 0);
                    if ( fErr )
                    	break;
				} else
					first_time = false;
					
            	SetArchiveStatus ( kDearchiving );
				// *** SEND NEXTFRAME (remember to set fErr)
				
            	CleanUpMemory (); // clean up all memory allocations for frame items (except nextFrame)
				// *** DISPOSE FDIL OBJECT
				
            	if ( !fErr ) {
	                SetArchiveStatus ( kConfirming );
	                length = 4;
	                ReadFromPipe ( response, &length, 0 );
	                response[length] = '\0'; // make it a C string for convenience
	                if ( strcmp ( "Abrt", response ) == 0 )
						fErr = kNewtAbortError;
	                else if ( strcmp ( "Conf", response ) != 0 )  // if not "Conf" protocol is messed up!
	                   	fErr = kDeskAbortError;
	                else
						;// *** CREATE FDIL OBJECT
	            }
	       	}
            CleanUpMemory (); // clean up all memory allocations for frame items (except nextFrame)
			// *** DISPOSE FDIL OBJECT
			if ( first_time || fErr < -28000 ) // we never got to FDput or HLFDIL Error - frame not sent
				TerminateDearchive ( );
		}
    } else
        fErr = kNewtAbortError; // we didn't get "Ready" so abort

    if ( fErr == EOF ) // normal end
    	fErr = kCommErrNoErr;
    if ( fErr )
        SetArchiveStatus ( kAborting );
    else
        SetArchiveStatus ( kEndArchive );
    EndArchive ( fErr, false );

    CDPipeDisconnect(ourPipe);
    return fErr;
}


/* DoConnectPipe
 *
 * Purpose: Try to connect to Newton PDA using currently selected methods.
 */
CommErr DoConnectPipe ( )
{
       CommErr         retn_value = kCommErrNoErr;
    long        endTime = 0;

    // ourPipe is a global defined in DILArchive.c
    retn_value = CDPipeListen(ourPipe, Timeout, 0, 0);

#ifndef _windows
    CHECKDILERROR((short) CDPipeAccept(ourPipe));
#else
    // This code doesn't need to be done on the MacOS side, but is currently required
    // for Windows.  You need to loop a little bit for the connection state to be
    //set to kCDIL_ConnectPending.  It might not do that if there's nobody on the
    // other side of the cable, though!

    // Loop until the Newton device connects
    endTime = CurrentTimeInSeconds() + kTimeoutInSecs;  // time at which we should stop looping
    while (CurrentTimeInSeconds() < endTime) {
        if (CDGetPipeState(ourPipe) == kCDIL_ConnectPending) {
            CHECKDILERROR((short) CDPipeAccept(ourPipe));
            break;
        } else
            CDIdle(ourPipe);
    }

    if (CurrentTimeInSeconds() >= endTime) {    // did we time out?
        CDPipeDisconnect(ourPipe) ;
        retn_value =  kOurTimeoutError;
    } else
       retn_value =  kCommErrNoErr;
#endif

       return retn_value;
}


/* ReadFromPipe
 *
 * Purpose: try to read 'length' bytes from pipe and pack them into 'response' string
 *          Purpose of this routine is to wait for data to arrive and while waiting keep
 *          event processing going. This keeps desktop machine from "freezing".
 *          Macintosh version calls routine EventLoop to check for and handle events.
 *          Since we may not get all the bytes in a single read, keep track of bytes read and
 *          keep trying to get all requested bytes until we time out.
 */
CommErr ReadFromPipe ( char *response, long *length, long swapping )
{
    Boolean     eom;
    short       num_tries=0;
    CommErr     readErr;
    CDIL_State  state;
    long        read_len, length_read=0;

    // *** Students write this
    state = CDGetPipeState(ourPipe);
    if (state != kCDIL_Connected && state != kCDIL_Busy )
        return kPipeNotReady;

    readErr = kTimeout;
    while ( (*length-length_read) && (num_tries++<=kNumRetries) && (readErr==kTimeout) )
    // this translates to: we haven't received total requested bytes,
    // haven't given up trying and
    // haven't failed a read because of something other than a timeout
    {
        read_len = *length-length_read; // necessary because if we timed out we read 0 bytes so we must reset it
		readErr = CDPipeRead ( ourPipe, (void *) response, &read_len, &eom, swapping, Encoding, Timeout, 0, 0 );
        if ( read_len != 0 ) {
            response += read_len;
            length_read += read_len;
        }
        EventLoop(); // if you don't need to service events, make this a do-nothing routine
    }
    if ( readErr == kCommErrNoErr )
        if ( *length != length_read )
            readErr = kUnknownError;
    return readErr;
}


/* WriteToPipe
 *
 * Purpose: try to write 'length' bytes to pipe from 'message' string
 *          Purpose of this routine is to encapsulate writing. 'swapping'
 *                     controls whether big endian/little endian byte swapping needs
 *                     happen
 */
CommErr WriteToPipe ( char *message, long *length, long swapping )
{
       return ( CDPipeWrite(ourPipe, (void *)message, length, true, swapping, Encoding, Timeout, 0, 0) );
}


/* ErrorStrings
 *
 * Purpose: return error string based on error number from the CDIL or FDIL
 */
char* ErrorStrings(CommErr theErr, char* theString)
{
    switch (theErr)
        {
        case kOurUserCancelled:  strcpy((char*) theString,
            (char*) "Archive Cancelled"); break;
        case kOurTimeoutError:  strcpy((char*) theString,
            (char*) "Timeout error. (determined by Archive)"); break;
        case kDeskAbortError:   strcpy((char*) theString,
            (char*) "Protocol error. (determined by Archive)"); break;
        case kNewtAbortError:   strcpy((char*) theString,
            (char*) "Abort error. (determined by Newton)"); break;
        case kUnknownError: strcpy((char*) theString,
            (char*) "Unknown OS error. (determined by Archive)"); break;
        case -97:  strcpy((char*) theString,
            (char*) "The port is busy. Quit the relevant application or restart if necessary (from ATalk Driver)");  break;

        // CDIL errors (see DILCPipe.h)
        case -28701:  strcpy((char*) theString,
            (char*) "Error on memory allocation.");  break;
        case -28702:  strcpy((char*) theString,
            (char*) "DIL pipe was set to a bad state.");  break;
        case -28703:  strcpy((char*) theString,
            (char*) "An unknown exception has occurred.");  break;
        case -28704:  strcpy((char*) theString,
            (char*) "The queue of asynchronous calls is full.");  break;
        case -28705:  strcpy((char*) theString,
            (char*) "Pipe has not been initialized.");  break;
        case -28706:  strcpy((char*) theString,
            (char*) "Parameter passed in was invalid.");  break;
        case -28707:  strcpy((char*) theString,
            (char*) "Pipe is not ready for operation.");  break;
        case -28708:  strcpy((char*) theString,
            (char*) "Timeout during DIL operation.");  break;

        // FDIL errors (see HLFDIL.h)
        case -28801:  strcpy((char*) theString,
            (char*) "Out of heap memory");  break;
        case -28802:  strcpy((char*) theString,
  (char*) "Out of temporary or other memory");  break;
        case -28803:  strcpy((char*) theString,
            (char*) "Unknown slot");  break;
        case -28804:  strcpy((char*) theString,
            (char*) "Slot size exceeded");  break;
        case -28805:  strcpy((char*) theString,
            (char*) "Slot size is required");  break;
        case -28806:  strcpy((char*) theString,
            (char*) "Unexpected data type");  break;

        // other errors
        case -28003:  strcpy((char*) theString,
            (char*) "The communication operation was aborted.");  break;
        case -28009:  strcpy((char*) theString,
            (char*) "Bad connection detected");  break;
        case -28017:  strcpy((char*) theString,
            (char*) "Out of memory");  break;
        case -28029:  strcpy((char*) theString,
            (char*) "Cannot connect to modem; no response.");  break;
        case -28030:  strcpy((char*) theString,
            (char*) "Disconnection detected.");  break;
        case -28100:  strcpy((char*) theString,
            (char*) "Disconnect occurred while reading.");  break;
        case -28101:  strcpy((char*) theString,
            (char*) "An error occurred while reading.");  break;
        case -28102:  strcpy((char*) theString,
            (char*) "The communication tool was not found");  break;
        case -28103:  strcpy((char*) theString,
            (char*) "Bad modem tool version");  break;
        default: sprintf(theString, "%ld", theErr);
        }
    return (char*) theString;
}

/* CreateArchiveFile
 *
 * Purpose: Using UNIX file I/O create and open a file and mark it as being for archiving strings or frames
 */
CommErr CreateArchiveFile( short type )
{
    CommErr err=0;
    char    name[256];

    // note that program doesn't know about file errors so we just return kUnknownError
    err = CreateFileName ( name ); // this is in file DILArchive.c as it is platform specific
    if ( !err ) {
        gArchFile = fopen ( name, "wb+" ); // create binary file as string files have weird limits
        if ( !gArchFile )
            err = kUnknownError;
        else {
            if ( type == kStrings )
                fputs ( "ARKS", gArchFile );
            else
                fputs ( "ARKF", gArchFile );
        }
    } else
        err = kUnknownError;
    return err;
}


/* OpenArchiveFile
 *
 * Purpose: Open existing archive file using the UNIX file I/O library and read whether it is for strings or frames
 */
CommErr OpenArchiveFile()
{
    char            name[256]; // 256 in case path information is included
    CommErr         err=kCommErrNoErr;
    int                             converted;

    // note that program doesn't know about file errors so we just return kUnknownError
    err = GetFileName ( name ); // this is in file DILArchive.c as it is platform specific
    if ( !err ) {
        gArchFile = fopen( name, "rb" );
        if ( !gArchFile )
            err = kUnknownError;
        else {
                        converted = fscanf ( gArchFile, "%4s", name );
            if ( converted != EOF && converted != 0) { // file begins with type of archive (strings or frames)
                if ( strcmp ( name, "ARKS" ) == 0 )
                    gXferMode = kStrings;
                else if ( strcmp ( name, "ARKF" ) == 0 )
                    gXferMode = kFrames;
                else {
                    err = kUnknownError; // file not one of ours.
                    fclose ( gArchFile );
                    gArchFile = 0;
                }
            } else
                err = kUnknownError;
        }
    } else
        err = kUnknownError; // user cancelled
    return err;
}


/* CreateArchiveFile
 *
 * Purpose: Using UNIX file I/O create and open a file and mark it as being for archiving strings or frames
 */
CommErr ArchiveString ( char *from_str )
{
    short   length;
    CommErr err=0;

    // we're going to assume we're at the right place to write a string
    length = strlen ( from_str );
    if ( fprintf( gArchFile, "%4hd%s", length, from_str ) == EOF )
        err = kUnknownError;

    return err;
}


/* DearchiveString
 *
 * Purpose: Using UNIX file I/O read first/next string from file and return to caller as 'to_str'
 */
CommErr DearchiveString ( char *to_str )
{
    int   length;
    CommErr err;

    // we're going to assume we're at the right place to get a string
    if ( fscanf ( gArchFile, "%4d", &length ) == EOF )
        return EOF;
    err = (CommErr)fgets ( to_str, length+1, gArchFile ); // length+1 because fgets read n-1 chars from stream
    if ( !err ) { // gets returns NULL if read failed
        if ( feof(gArchFile) )
            err = EOF;
        else
            err = kUnknownError;
    } else
        err = kCommErrNoErr;
    return err;
}



/* EndArchive
 *
 * Purpose: Terminate archiving as gracefully as possible including closing file and notifying Newton PDA of end.
 */
CommErr EndArchive ( CommErr endState, Boolean sending )
{
    long    length;    char    response[256];

    SetArchiveStatus ( endState );
    if ( endState  )
    {
        length = 6L; // send terminator character
        WriteToPipe ("ABORT\4", &length, 0);
    } else { // no error
        if ( !sending )
        {
            length = 12L; // send null terminator too (changed to \4 instead)
            WriteToPipe ("END ARCHIVE\4", &length, 0);
        }
    }

    length = 4;
    ReadFromPipe ( response, &length, 0 ); // wait for acknowledgement we're ending

    if ( gArchFile ) {
        fclose ( gArchFile );
        gArchFile = 0;
    }

    return endState;
}



/* DearchiveFrame
 *
 * Purpose: Using UNIX file I/O read first/next frame from file, bind it to the
 * object passed in and return to caller
 */
CommErr DearchiveFrame ( DILObj *cur_obj, Boolean isArray )
{
	CommErr	fErr;
	char 	temp_str[256], name[256], *oclass_ptr, oclass[256], *name_ptr, *temp_ptr;
	short	var_type;
	long	max_len, cur_len;
	void	*bind_obj; // object being bound

    do { 
    	if ( !isArray ) {
	    	fErr = fscanf ( gArchFile, "%256s %256s ", name, temp_str ); // name is the slot name, temp_str is used here for var type of entry
			name_ptr = name;
		} else {
			name_ptr = NULL;
	    	fErr = fscanf ( gArchFile, "%256s ", temp_str ); // array entries have no names, temp_str is used here for var type of entry
		}
	    if ( fErr == EOF )
	    	return fErr; // we're done!

		// "null out" oclass as it is not necessarily set 
	    oclass_ptr = NULL;
	    	
	    // can't use switch for following as we used strings for type for easy reading
	    if ( strcmp ( temp_str, "CHAR" ) == 0 ) {
	    	char *the_char = (char *)AllocateHeapObject ( sizeof ( char ) ); // boy does this feel silly...
			*gCurMemItem++ = (heap_ptr)the_char;
	    	*the_char = fgetc ( gArchFile );
	    	max_len = sizeof ( char );
	    	cur_len = max_len;
	    	bind_obj = the_char;
	    	var_type = kDILCharacter;
	    } else if ( strcmp ( temp_str, "UCHR" ) == 0 ) {
	    	short *the_uchar = (short *)AllocateHeapObject ( sizeof ( short ) );
			*gCurMemItem++ = (heap_ptr)the_uchar;
	    	fscanf ( gArchFile, "%d", the_uchar );
	    	max_len = sizeof ( short );
	    	cur_len = max_len;
	    	bind_obj = the_uchar;
	    	var_type = kDILUnicodeCharacter;
	    } else if ( strcmp ( temp_str, "STRN" ) == 0 ) {
	    	char *heap_str;
	    	fpos_t	pos;
	    	
	    	fgetpos ( gArchFile, &pos ); // get current position before looking for class
	    	fscanf ( gArchFile, "%256s", temp_str );
	    	if ( strcmp ( temp_str, "class" ) == 0 ) { // string has a class associated with it
	    		fscanf ( gArchFile, "%256s", oclass ); // get class
	    		oclass_ptr = oclass;
			} else {
				if ( name_ptr == NULL && oclass_ptr == NULL ) { // string is in an array, the following fixes a bug
					strcpy ( oclass, "string" );
					oclass_ptr = oclass;
				}
				fsetpos ( gArchFile, &pos); // reset to where we were before trying to get class
			}
			
			while ( fgetc(gArchFile) != '"' ) // move past opening "
				;
			fscanf ( gArchFile, "%[^\"]", temp_str ); // better not be " char in string	
			fgetc( gArchFile ) ; // no past closing "  	
	    	max_len = strlen( temp_str ); 
			cur_len = -1;
	    	heap_str = (char *)AllocateHeapObject ( max_len+1 ); // +1 for null char at end
			*gCurMemItem++ = (heap_ptr)heap_str;
			strcpy ( heap_str, temp_str );
	    	
	    	bind_obj = heap_str;
	    	var_type = kDILString;
	    } else if ( strcmp ( temp_str, "BOOL" ) == 0 ) {
	    	Boolean *heap_bool = (Boolean *)AllocateHeapObject ( sizeof ( Boolean ) );
			*gCurMemItem++ = (heap_ptr)heap_bool;
			fscanf ( gArchFile, "%256s", temp_str );
			if ( strcmp ( temp_str, "TRUE" ) == 0 )
				*heap_bool = true;
			else
				*heap_bool = false;
			max_len = sizeof ( Boolean );
			cur_len = max_len;
	    	bind_obj = heap_bool;
			var_type = kDILBoolean;
	    } else if ( strcmp ( temp_str, "LONG" ) == 0 ) {
	    	long *the_long = (long *)AllocateHeapObject ( sizeof ( long ) );
			*gCurMemItem++ = (heap_ptr)the_long;
	    	fscanf ( gArchFile, "%ld", the_long );
	    	max_len = sizeof ( long );
			cur_len = max_len;
	    	bind_obj = the_long;
	    	var_type = kDILInteger;
	   	} else if ( strcmp ( temp_str, "IMMD" ) == 0 ) {
	    	long *the_immediate = (long *)AllocateHeapObject ( sizeof ( long ) );
			*gCurMemItem++ = (heap_ptr)the_immediate;
	    	fscanf ( gArchFile, "%ld", the_immediate );
	    	max_len = sizeof ( long );	   	
			cur_len = max_len;
	    	bind_obj = the_immediate;
	    	var_type = kDILImmediate;
		} else if ( strcmp ( temp_str, "PARR" ) == 0 ) {
			DILObj *next_obj;
			// *** CREATE AN FDIL OBJECT
			*gCurDILObj++ = next_obj;
			fErr = DearchiveFrame ( next_obj, true );
			if ( fErr )
				break;
	    	bind_obj = next_obj;
	    	max_len = 0; // correct for arrays and frames
	    	cur_len = -1;
	    	var_type = kDILArray;
	    } else if ( strcmp ( temp_str, "NARR" ) == 0 ) {
			DILObj *next_obj;
			// *** CREATE AN FDIL OBJECT
			*gCurDILObj++ = next_obj;
			fscanf ( gArchFile, "%256s", temp_str );
			if ( strcmp(temp_str, "class" ) == 0 ) {
				fscanf ( gArchFile, "%256s", oclass );
				oclass_ptr = oclass;
				fErr = DearchiveFrame ( next_obj, true );
		    	bind_obj = next_obj;
		    	max_len = 0; // correct for arrays and frames
		    	cur_len = -1;
		    	var_type = kDILArray;
		    } else {
		    	fErr = kDeskAbortError;
		    	break;
		    }
	    } else if ( strcmp ( temp_str, "FRAM" ) == 0 ) {
			DILObj *next_obj;
			// *** CREATE AN FDIL OBJECT
			*gCurDILObj++ = next_obj;
			fErr = DearchiveFrame ( next_obj, false );
	    	bind_obj = next_obj;
	    	max_len = 0; // correct for arrays and frames
	    	cur_len = -1;
	    	var_type = kDILFrame;	
	    } else if ( strcmp ( temp_str, "SREC" ) == 0 ) {
	    	long *the_long = (long *)AllocateHeapObject ( sizeof ( long ) );
			*gCurMemItem++ = (heap_ptr)the_long;
	    	fscanf ( gArchFile, "%ld\n", the_long );
	    	max_len = sizeof ( long );
			cur_len = max_len;
	    	bind_obj = the_long;
	    	var_type = kDILSmallRect;
	    } else if ( strcmp ( temp_str, "PRCD" ) == 0 ) {
	    	; // not documented
	    } else if ( strcmp ( temp_str, "SYMB" ) == 0 ) {
	    	char *the_str;
			fscanf ( gArchFile, "%256s", temp_str );
			max_len = strlen ( temp_str );
			cur_len = max_len;
			the_str = (char *)AllocateHeapObject ( max_len+1 ); // +1 for null at end of string
			*gCurMemItem++ = (heap_ptr)the_str;
			strcpy ( the_str, temp_str );
			bind_obj = the_str;
			var_type = kDILSymbol;
	    } else if ( strcmp ( temp_str, "BINR" ) == 0 ) {
	    	bind_obj = 0;
			fscanf ( gArchFile, "%256s", temp_str );
			if ( strcmp ( temp_str, "class" ) == 0 ) {
				fscanf ( gArchFile, "%256s", oclass );
				if ( strcmp ( oclass, "Real" ) == 0 ) {
	    			double *the_double;
	    			
	    			oclass_ptr = oclass;
					the_double = (double *)AllocateHeapObject ( sizeof ( double ) );
					*gCurMemItem++ = (heap_ptr)the_double;
					fscanf ( gArchFile, "%lf", the_double );
					max_len = sizeof ( double );
					cur_len = max_len;
#ifdef _windows
						// ieee reals have bytes in the opposite order...
						unsigned long tmpWord1;
						unsigned long tmpWord2;
						tmpWord1 = ((unsigned long *)the_double)[0];
						tmpWord2 = ((unsigned long *)the_double)[1];
						tmpWord1 =  ((tmpWord1 & 0x000000ff) << 24) |
									((tmpWord1 & 0x0000ff00) <<  8) |
									((tmpWord1 & 0x00ff0000) >>  8) |
									((tmpWord1 & 0xff000000) >> 24);
						tmpWord2 =  ((tmpWord2 & 0x000000ff) << 24) |
									((tmpWord2 & 0x0000ff00) <<  8) |
									((tmpWord2 & 0x00ff0000) >>  8) |
									((tmpWord2 & 0xff000000) >> 24);
						((unsigned long *)&tmpDouble)[0] = tmpWord2;
						((unsigned long *)&tmpDouble)[1] = tmpWord1;
#endif
					bind_obj = the_double;
				} else
					oclass_ptr = oclass;
			} 
			if ( oclass_ptr != NULL && strcmp ( oclass, "Real" ) != 0 ) { // either different class or no class
				long	temp_len;
				
				fscanf ( gArchFile, "%ld ", &temp_len );
				max_len = temp_len;
				cur_len = max_len;
				temp_ptr = (char *)AllocateHeapObject ( max_len );
				*gCurMemItem++ = (heap_ptr)temp_ptr;
				bind_obj = temp_ptr;
				while ( temp_len ) {
					*temp_ptr++ = fgetc ( gArchFile );
					temp_len--; // because of a compiler error
				}
				fgetc ( gArchFile ); // push past ending '\n' char
			}
			var_type = kDILBinaryObject;
	    } else if ( strcmp ( temp_str, "NULL" ) == 0 ) {
			bind_obj = NULL;
			max_len = 0;
			cur_len = max_len;
			var_type = kDILNIL;
	    } else if ( strcmp ( temp_str, "BLOB" ) == 0 ) {
	    	continue; // unimplemented
	    } else if ( strcmp ( temp_str, "UNKN" ) == 0 )
			continue; // ignore this
			
		// *** BIND SLOT TO OBJECT (remember to set fErr)
		temp_str[0] = '\0';
    	fscanf ( gArchFile, "%2[\r\n\\]", temp_str );
    	if ( strchr(temp_str, '\\') ) // we found an end of object marker
    		break;
	} while ( !fErr );
	return fErr;
}



/* WriteFrameToFile
 *
 * Purpose: Take unbound frame data and write it to file
 */
objErr	WriteFrameToFile ( DILObj *theObject )
{
	objErr			fErr = kCommErrNoErr;
	slotDefinition *list = nil;
	long			len = 3;
	short			x;

	/*	We should have an unbound frame	list */
	// *** GET UNBOUND DATA LIST
	if ( !list )						/* if not, we got an empty soup frame?; fail! */
		return kDeskAbortError;
		
	/*	traverse list		*/
	// *** LOOP OVER THE LIST OF ENTRIES

	fputc ( '\\', gArchFile ); // frame separator is two backslash characters - substitute what you want

	// *** FREE UNBOUND DATA
	fflush(gArchFile);
	return fErr;	
}



/* PrintFrame
 *
 * Purpose: To output unbound data to file
 */
void PrintFrame ( slotDefinition *thisEntry )
{
	short y;
	
	PrintSlot( thisEntry );
	
	if (thisEntry->children) {
		// *** LOOP OVER THE LIST OF CHILDREN
		fputc ( '\\', gArchFile ); // end children
	}
}



/* PrintSlot
 *
 * Purpose: To output an entry from unbound data
 */
void	PrintSlot ( slotDefinition *thisEntry )
{	
	void	*the_data;
	short	the_data_type;
	char	*the_data_class, *the_slot_name;
	
	// *** INITIALIZE LOCAL VARIABLES
	
	if (the_slot_name)
		fprintf( gArchFile, "%s ", the_slot_name);
	
	switch (the_data_type)
	{
		case kDILCharacter:			/*	ASCII Character					*/
			fprintf( gArchFile, "CHAR %c\n", *(char *)the_data );
			break;
			
		case kDILUnicodeCharacter:	/*	Unicode Character				*/
			fprintf( gArchFile,  "UCHR %d\n", (short *)the_data);
			break;
			
		case kDILString:			/*	String (null-terminated)		*/
			fprintf( gArchFile, "STRN " );
			if (the_data_class)
				// there's a special class defined for this string; show it.
				fprintf( gArchFile, "class %s \"%s\"\n", (char *)the_data_class, (char *)the_data );
			else
				// generic string; no special class.
				fprintf( gArchFile,  "\"%s\"\n", (char *)the_data);
			break;
			
		case kDILBoolean:			/*	Boolean							*/
			fprintf( gArchFile, "BOOL " );
			if (*(Boolean *)the_data)
				fprintf( gArchFile,  "TRUE\n");
			else
				fprintf( gArchFile,  "FALSE\n");
			break;
			
		case kDILImmediate:			/*	Indeterminate Immediate type	*/
			fprintf( gArchFile, "IMMD %ld\n", *(long *)the_data);
			break;
			
		case kDILInteger:			/*	Integer (4 byte)				*/
			fprintf( gArchFile, "LONG %ld\n", *(long *)the_data);
			break;
			
		case kDILPlainArray:		/*	Anonymous Array					*/
			fprintf (gArchFile, "PARR \n");
			break;
			
		case kDILArray:				/*	Named Array						*/
			if (the_data_class)
				fprintf (gArchFile, "NARR class %s\n", the_data_class);
			else
				fprintf (gArchFile, "NARR class ???\n");
			break;
		
		case kDILFrame:				/*	Frame							*/
			fprintf (gArchFile, "FRAM \n");
			break;
		
		case kDILSmallRect:			/*	Small rect						*/
			fprintf (gArchFile, "SREC %ld\n", *(long *)the_data );
			break;
		
		case kDILPrecedent:			/*	Repeated Item					*/
			fprintf (gArchFile, "PRCD %s\n", (char *)the_data);
			break;
		
		case kDILSymbol:			/*	Object Symbol					*/
			fprintf( gArchFile, "SYMB %s\n", (char *)the_data);
			break;
		
		case kDILBinaryObject:		/*	Small Binary Object ( < 32K )	*/
			{
				unsigned long bin_len=0;
				unsigned char *bin_ptr;

				fprintf ( gArchFile, "BINR " );
				if (the_data_class)
					if (0==strcmp(the_data_class,"Real")) {
						double tmpDouble;
#ifdef _windows
						// ieee reals have bytes in the opposite order...
						unsigned long tmpWord1;
						unsigned long tmpWord2;
						tmpWord1 = ((unsigned long *)the_data)[0];
						tmpWord2 = ((unsigned long *)the_data)[1];
						tmpWord1 =  ((tmpWord1 & 0x000000ff) << 24) |
									((tmpWord1 & 0x0000ff00) <<  8) |
									((tmpWord1 & 0x00ff0000) >>  8) |
									((tmpWord1 & 0xff000000) >> 24);
						tmpWord2 =  ((tmpWord2 & 0x000000ff) << 24) |
									((tmpWord2 & 0x0000ff00) <<  8) |
									((tmpWord2 & 0x00ff0000) >>  8) |
									((tmpWord2 & 0xff000000) >> 24);
						((unsigned long *)&tmpDouble)[0] = tmpWord2;
						((unsigned long *)&tmpDouble)[1] = tmpWord1;
#else
						tmpDouble = *(double *)the_data;
#endif
						fprintf( gArchFile,  "class Real %f\n", tmpDouble);
					}
					else
						fprintf( gArchFile,  "class %s ", the_data_class );

				if ( strcmp ( the_data_class, "Real" ) != 0 )  { // some binary other than real
					bin_len = thisEntry->length;
					fprintf ( gArchFile, "%ld ", bin_len );
					for ( bin_ptr=(unsigned char *)the_data; bin_len; bin_len-- )
						fputc ( *bin_ptr++, gArchFile );
					fputc ( '\n', gArchFile ); // \n is end of entry
				}
			}
			break;
		
		case kDILNIL:				/*	nil object						*/
			fprintf (gArchFile, "NULL \n");
			break;
		
		case kDILBLOB:				/*	Large Binary Object				*/
		{
			// note: we'll assume it works just like a kDILBinaryObject as it isn't documented
			unsigned long bin_len=0;
			unsigned char *bin_ptr;
			
			fprintf (gArchFile, "BLOB \n"); 
			break;
		}
		
		default:					// ~~~~~ ignore these
			fprintf (gArchFile, "UNKN \n");
			break;
	}
}



/* CleanUpMemory
 *
 * Purpose: To clean up DILObjs and heap blocks allocated in binding a frame
 */
CommErr	CleanUpMemory ( )
{
	DILObj		**tempDILObj;
	heap_ptr	*genericMem;
	CommErr		fErr=kCommErrNoErr;
	
	// now dispose of all objects and memory associated with nextFrame
	tempDILObj = gDILObjArray;
	genericMem = gMemArray;
	
	while ( *tempDILObj ) {
		// *** DISPOSE FDIL OBJECT (remember to set fErr)
		*tempDILObj++ = (DILObj *)0; // null out entry in gDILObjArray
		if ( fErr )
			break;
	}
	
	while ( *genericMem ) {
		DisposeHeapObject ( *genericMem );
		*genericMem++ = (Ptr)0;
	}
	
	return fErr;
}


/* TerminateDearchive
 *
 * Purpose: To terminate dearchiving and send the Newt an error frame since it is waiting
 *          for a frame.
 */

void TerminateDearchive (  )
{
	DILObj *aFrame;
	Boolean flag=true;
	
	// *** CREATE AN FDIL OBJECT
	// *** BIND A TRUE TO THE __ARCHIVEERROR SLOT
	// *** SEND THE ERROR FRAME
	// *** DISPOSE OF THE FDIL OBJECT
}
                                                                                            v   v   R         O  Y      	&   
PROTOCOL.C   TEXTMACA     TEXTMACA                   :       $         J                                    G       Y  Y  P1             H 	Monaco                             mu mu'  
Z  
\             v   v   R#
P    R MPSR  MWBB   *           "       n                                                            